<?php
/**
 * LexiPro - Funciones de Autenticación
 */

require_once __DIR__ . '/../config/database.php';

class Auth {
    private $db;
    
    public function __construct() {
        $this->db = Database::getInstance();
    }
    
    /**
     * Iniciar sesión
     */
    public function login($usuario, $password) {
        $user = $this->db->fetchOne(
            "SELECT u.*, r.nombre as rol_nombre, r.nivel as rol_nivel 
             FROM usuarios u 
             INNER JOIN roles r ON u.rol_id = r.id 
             WHERE u.usuario = ? AND u.activo = 1",
            [$usuario]
        );
        
        if ($user && password_verify($password, $user['password'])) {
            // Actualizar último acceso
            $this->db->update('usuarios', 
                ['ultimo_acceso' => date('Y-m-d H:i:s')],
                'id = ?',
                [$user['id']]
            );
            
            // Guardar en sesión
            $_SESSION['user_id'] = $user['id'];
            $_SESSION['user_nombre'] = $user['nombre'];
            $_SESSION['user_apellido'] = $user['apellido'];
            $_SESSION['user_email'] = $user['email'];
            $_SESSION['user_foto'] = $user['foto'];
            $_SESSION['user_rol_id'] = $user['rol_id'];
            $_SESSION['user_rol_nombre'] = $user['rol_nombre'];
            $_SESSION['user_rol_nivel'] = $user['rol_nivel'];
            $_SESSION['logged_in'] = true;
            
            return true;
        }
        
        return false;
    }
    
    /**
     * Cerrar sesión
     */
    public function logout() {
        session_destroy();
        header('Location: /lexipro/login.php');
        exit;
    }
    
    /**
     * Verificar si está autenticado
     */
    public function isLoggedIn() {
        return isset($_SESSION['logged_in']) && $_SESSION['logged_in'] === true;
    }
    
    /**
     * Requerir autenticación
     */
    public function requireLogin() {
        if (!$this->isLoggedIn()) {
            header('Location: /lexipro/login.php');
            exit;
        }
    }
    
    /**
     * Obtener usuario actual
     */
    public function getCurrentUser() {
        if (!$this->isLoggedIn()) {
            return null;
        }
        
        return [
            'id' => $_SESSION['user_id'],
            'nombre' => $_SESSION['user_nombre'],
            'apellido' => $_SESSION['user_apellido'],
            'email' => $_SESSION['user_email'],
            'foto' => $_SESSION['user_foto'],
            'rol_id' => $_SESSION['user_rol_id'],
            'rol_nombre' => $_SESSION['user_rol_nombre'],
            'rol_nivel' => $_SESSION['user_rol_nivel']
        ];
    }
    
    /**
     * Verificar permiso
     */
    public function hasPermission($modulo, $accion = 'ver') {
        if (!$this->isLoggedIn()) {
            return false;
        }
        
        // Administrador tiene todos los permisos
        if ($_SESSION['user_rol_nivel'] >= 100) {
            return true;
        }
        
        $permiso = $this->db->fetchOne(
            "SELECT {$accion} FROM permisos WHERE rol_id = ? AND modulo = ?",
            [$_SESSION['user_rol_id'], $modulo]
        );
        
        return $permiso && $permiso[$accion] == 1;
    }
    
    /**
     * Requerir permiso
     */
    public function requirePermission($modulo, $accion = 'ver') {
        if (!$this->hasPermission($modulo, $accion)) {
            http_response_code(403);
            die('Acceso denegado');
        }
    }
}
