/**
 * LexiPro - JavaScript Global
 */

// Funciones auxiliares
const LexiPro = {
    // Formatear fecha
    formatDate(date) {
        if (!date) return '';
        const d = new Date(date);
        return d.toLocaleDateString('es-PY');
    },

    // Formatear fecha y hora
    formatDateTime(datetime) {
        if (!datetime) return '';
        const d = new Date(datetime);
        return d.toLocaleString('es-PY');
    },

    // Formatear moneda
    formatMoney(amount) {
        return 'Gs. ' + new Intl.NumberFormat('es-PY').format(amount);
    },

    // Mostrar alerta
    showAlert(message, type = 'info') {
        const alertDiv = document.createElement('div');
        alertDiv.className = `alert alert-${type}`;
        alertDiv.innerHTML = `
            <span>${this.getAlertIcon(type)}</span>
            <span>${message}</span>
        `;

        document.body.insertBefore(alertDiv, document.body.firstChild);

        setTimeout(() => {
            alertDiv.remove();
        }, 5000);
    },

    // Obtener icono de alerta
    getAlertIcon(type) {
        const icons = {
            success: '✓',
            danger: '✗',
            warning: '⚠',
            info: 'ℹ'
        };
        return icons[type] || icons.info;
    },

    // Confirmar acción
    confirm(message) {
        return window.confirm(message);
    },

    // Hacer petición fetch
    async fetch(url, options = {}) {
        try {
            const response = await fetch(url, {
                ...options,
                headers: {
                    'Content-Type': 'application/json',
                    ...options.headers
                }
            });

            const data = await response.json();
            return data;
        } catch (error) {
            console.error('Error en petición:', error);
            throw error;
        }
    },

    // Debounce
    debounce(func, wait) {
        let timeout;
        return function executedFunction(...args) {
            const later = () => {
                clearTimeout(timeout);
                func(...args);
            };
            clearTimeout(timeout);
            timeout = setTimeout(later, wait);
        };
    },

    // Validar email
    validateEmail(email) {
        const re = /^[^\s@]+@[^\s@]+\.[^\s@]+$/;
        return re.test(email);
    },

    // Validar RUC Paraguay
    validateRUC(ruc) {
        const re = /^\d{6,8}-\d{1}$/;
        return re.test(ruc);
    },

    // Validar Cédula Paraguay
    validateCedula(cedula) {
        const cleaned = cedula.replace(/\./g, '');
        const re = /^\d{6,8}$/;
        return re.test(cleaned);
    },

    // Abrir modal
    openModal(modalId) {
        const modal = document.getElementById(modalId);
        if (modal) {
            modal.classList.add('active');
        }
    },

    // Cerrar modal
    closeModal(modalId) {
        const modal = document.getElementById(modalId);
        if (modal) {
            modal.classList.remove('active');
        }
    },

    // Serializar formulario
    serializeForm(form) {
        const formData = new FormData(form);
        const data = {};

        for (let [key, value] of formData.entries()) {
            // Manejar checkboxes
            if (form.elements[key].type === 'checkbox') {
                data[key] = form.elements[key].checked ? 1 : 0;
            } else {
                data[key] = value;
            }
        }

        return data;
    },

    // Cargar imagen preview
    previewImage(input, previewId) {
        if (input.files && input.files[0]) {
            const reader = new FileReader();

            reader.onload = function (e) {
                const preview = document.getElementById(previewId);
                if (preview) {
                    preview.src = e.target.result;
                }
            };

            reader.readAsDataURL(input.files[0]);
        }
    },

    // Cambiar tema (Claro/Oscuro)
    toggleTheme() {
        const body = document.body;
        const isDark = body.classList.toggle('dark-mode');
        localStorage.setItem('theme', isDark ? 'dark' : 'light');

        // Actualizar icono
        const icon = document.getElementById('themeIcon');
        if (icon) {
            icon.textContent = isDark ? '☀️' : '🌙';
        }
    },

    // Inicializar tema
    initTheme() {
        const savedTheme = localStorage.getItem('theme');
        const icon = document.getElementById('themeIcon');

        if (savedTheme === 'dark') {
            document.body.classList.add('dark-mode');
            if (icon) icon.textContent = '☀️';
        } else {
            if (icon) icon.textContent = '🌙';
        }
    }
};

// Inicializar tema al cargar
document.addEventListener('DOMContentLoaded', () => {
    LexiPro.initTheme();
});

// Cerrar modales al hacer clic fuera
document.addEventListener('click', (e) => {
    if (e.target.classList.contains('modal')) {
        e.target.classList.remove('active');
    }
});

// Prevenir envío de formularios con Enter (excepto en textareas)
document.addEventListener('keydown', (e) => {
    if (e.key === 'Enter' && e.target.tagName !== 'TEXTAREA') {
        const form = e.target.closest('form');
        if (form && e.target.type !== 'submit') {
            e.preventDefault();
        }
    }
});

// Exportar para uso global
window.LexiPro = LexiPro;
