/**
 * LexiPro - Widget de Chat Web (Flotante)
 * Este archivo debe ser incluido en index.html
 */

(function () {
    'use strict';

    // Generar o recuperar ID de sesión
    function getSessionId() {
        let sessionId = localStorage.getItem('lexipro_chat_session');
        if (!sessionId) {
            sessionId = 'session_' + Date.now() + '_' + Math.random().toString(36).substr(2, 9);
            localStorage.setItem('lexipro_chat_session', sessionId);
        }
        return sessionId;
    }

    // Obtener datos del usuario
    function getUserData() {
        const data = localStorage.getItem('lexipro_chat_user');
        return data ? JSON.parse(data) : null;
    }

    // Guardar datos del usuario
    function saveUserData(nombre, apellido, email) {
        const data = { nombre, apellido, email };
        localStorage.setItem('lexipro_chat_user', JSON.stringify(data));
        return data;
    }

    // Crear estructura HTML del chat
    function createChatWidget() {
        const widget = document.createElement('div');
        widget.id = 'lexipro-chat-widget';
        widget.innerHTML = `
            <style>
                #lexipro-chat-widget {
                    position: fixed;
                    bottom: 20px;
                    right: 20px;
                    z-index: 9999;
                    font-family: 'Segoe UI', Tahoma, Geneva, Verdana, sans-serif;
                }
                
                .chat-button {
                    width: 60px;
                    height: 60px;
                    border-radius: 50%;
                    background: linear-gradient(135deg, #1e3c72 0%, #2a5298 100%);
                    color: white;
                    border: none;
                    cursor: pointer;
                    box-shadow: 0 4px 12px rgba(0,0,0,0.3);
                    display: flex;
                    align-items: center;
                    justify-content: center;
                    font-size: 28px;
                    transition: all 0.3s;
                }
                
                .chat-button:hover {
                    transform: scale(1.1);
                    box-shadow: 0 6px 16px rgba(0,0,0,0.4);
                }
                
                .chat-button.active {
                    background: #ef4444;
                }
                
                .chat-badge {
                    position: absolute;
                    top: -5px;
                    right: -5px;
                    background: #ef4444;
                    color: white;
                    border-radius: 50%;
                    width: 24px;
                    height: 24px;
                    display: flex;
                    align-items: center;
                    justify-content: center;
                    font-size: 12px;
                    font-weight: bold;
                    display: none;
                }
                
                .chat-badge.show {
                    display: flex;
                }
                
                .chat-window {
                    position: absolute;
                    bottom: 80px;
                    right: 0;
                    width: 350px;
                    height: 500px;
                    background: white;
                    border-radius: 16px;
                    box-shadow: 0 8px 32px rgba(0,0,0,0.2);
                    display: none;
                    flex-direction: column;
                    overflow: hidden;
                }
                
                .chat-window.show {
                    display: flex;
                }
                
                .chat-header {
                    background: linear-gradient(135deg, #1e3c72 0%, #2a5298 100%);
                    color: white;
                    padding: 20px;
                    display: flex;
                    justify-content: space-between;
                    align-items: center;
                }
                
                .chat-header h3 {
                    margin: 0;
                    font-size: 18px;
                }
                
                .chat-close {
                    background: none;
                    border: none;
                    color: white;
                    font-size: 24px;
                    cursor: pointer;
                    padding: 0;
                    width: 30px;
                    height: 30px;
                }
                
                .chat-messages {
                    flex: 1;
                    overflow-y: auto;
                    padding: 20px;
                    background: #f9fafb;
                }
                
                .chat-message {
                    margin-bottom: 15px;
                    display: flex;
                    flex-direction: column;
                }
                
                .chat-message.user {
                    align-items: flex-end;
                }
                
                .chat-message.agent {
                    align-items: flex-start;
                }
                
                .message-bubble {
                    max-width: 70%;
                    padding: 12px 16px;
                    border-radius: 16px;
                    word-wrap: break-word;
                }
                
                .chat-message.user .message-bubble {
                    background: #2a5298;
                    color: white;
                    border-bottom-right-radius: 4px;
                }
                
                .chat-message.agent .message-bubble {
                    background: white;
                    color: #1f2937;
                    border-bottom-left-radius: 4px;
                    box-shadow: 0 2px 4px rgba(0,0,0,0.1);
                }
                
                .message-time {
                    font-size: 11px;
                    color: #9ca3af;
                    margin-top: 4px;
                }
                
                .chat-form {
                    padding: 15px;
                    background: white;
                    border-top: 1px solid #e5e7eb;
                }
                
                .chat-input {
                    width: 100%;
                    padding: 12px;
                    border: 2px solid #e5e7eb;
                    border-radius: 24px;
                    font-size: 14px;
                    outline: none;
                    resize: none;
                    font-family: inherit;
                }
                
                .chat-input:focus {
                    border-color: #2a5298;
                }
                
                .chat-send {
                    background: #2a5298;
                    color: white;
                    border: none;
                    padding: 10px 20px;
                    border-radius: 20px;
                    cursor: pointer;
                    font-weight: 600;
                    margin-top: 10px;
                    width: 100%;
                }
                
                .chat-send:hover {
                    background: #1e3c72;
                }
                
                .chat-register {
                    padding: 20px;
                }
                
                .chat-register input {
                    width: 100%;
                    padding: 12px;
                    margin-bottom: 10px;
                    border: 2px solid #e5e7eb;
                    border-radius: 8px;
                    font-size: 14px;
                }
                
                .typing-indicator {
                    display: none;
                    padding: 10px;
                    color: #6b7280;
                    font-size: 13px;
                    font-style: italic;
                }
                
                .typing-indicator.show {
                    display: block;
                }
                
                @media (max-width: 480px) {
                    .chat-window {
                        width: calc(100vw - 40px);
                        height: calc(100vh - 100px);
                    }
                }
            </style>
            
            <button class="chat-button" id="chatButton">
                💬
                <span class="chat-badge" id="chatBadge">0</span>
            </button>
            
            <div class="chat-window" id="chatWindow">
                <div class="chat-header">
                    <h3>⚖️ LexiPro - Consultas</h3>
                    <button class="chat-close" id="chatClose">×</button>
                </div>
                
                <div id="chatRegister" class="chat-register" style="display: none;">
                    <h4 style="margin-bottom: 15px;">Bienvenido</h4>
                    <p style="margin-bottom: 15px; color: #6b7280; font-size: 14px;">
                        Para comenzar, por favor ingresa tus datos:
                    </p>
                    <input type="text" id="chatNombre" placeholder="Nombre" required>
                    <input type="text" id="chatApellido" placeholder="Apellido" required>
                    <input type="email" id="chatEmail" placeholder="Email" required>
                    <button class="chat-send" id="chatStartBtn">Iniciar Chat</button>
                </div>
                
                <div id="chatContent" style="display: none; flex: 1; display: flex; flex-direction: column;">
                    <div class="chat-messages" id="chatMessages"></div>
                    <div class="typing-indicator" id="typingIndicator">El equipo está escribiendo...</div>
                    <div class="chat-form">
                        <textarea class="chat-input" id="chatInput" placeholder="Escribe tu mensaje..." rows="1"></textarea>
                        <button class="chat-send" id="chatSendBtn">Enviar</button>
                    </div>
                </div>
            </div>
        `;

        document.body.appendChild(widget);

        // Event listeners
        const chatButton = document.getElementById('chatButton');
        const chatWindow = document.getElementById('chatWindow');
        const chatClose = document.getElementById('chatClose');
        const chatStartBtn = document.getElementById('chatStartBtn');
        const chatSendBtn = document.getElementById('chatSendBtn');
        const chatInput = document.getElementById('chatInput');

        chatButton.addEventListener('click', toggleChat);
        chatClose.addEventListener('click', closeChat);
        chatStartBtn.addEventListener('click', startChat);
        chatSendBtn.addEventListener('click', sendMessage);
        chatInput.addEventListener('keypress', (e) => {
            if (e.key === 'Enter' && !e.shiftKey) {
                e.preventDefault();
                sendMessage();
            }
        });

        // Verificar si ya tiene datos guardados
        checkUserData();

        // Cargar mensajes cada 5 segundos
        setInterval(loadMessages, 5000);
    }

    function toggleChat() {
        const chatWindow = document.getElementById('chatWindow');
        const chatButton = document.getElementById('chatButton');
        chatWindow.classList.toggle('show');
        chatButton.classList.toggle('active');

        if (chatWindow.classList.contains('show')) {
            loadMessages();
        }
    }

    function closeChat() {
        const chatWindow = document.getElementById('chatWindow');
        const chatButton = document.getElementById('chatButton');
        chatWindow.classList.remove('show');
        chatButton.classList.remove('active');
    }

    function checkUserData() {
        const userData = getUserData();
        if (userData) {
            showChatContent();
            loadMessages();
        } else {
            showRegisterForm();
        }
    }

    function showRegisterForm() {
        document.getElementById('chatRegister').style.display = 'block';
        document.getElementById('chatContent').style.display = 'none';
    }

    function showChatContent() {
        document.getElementById('chatRegister').style.display = 'none';
        document.getElementById('chatContent').style.display = 'flex';
    }

    function startChat() {
        const nombre = document.getElementById('chatNombre').value.trim();
        const apellido = document.getElementById('chatApellido').value.trim();
        const email = document.getElementById('chatEmail').value.trim();

        if (!nombre || !apellido || !email) {
            alert('Por favor completa todos los campos');
            return;
        }

        if (!validateEmail(email)) {
            alert('Por favor ingresa un email válido');
            return;
        }

        saveUserData(nombre, apellido, email);
        showChatContent();

        // Enviar mensaje de bienvenida automático
        sendMessage('Hola, necesito información', true);
    }

    async function sendMessage(customMessage = null, isAuto = false) {
        const userData = getUserData();
        if (!userData) return;

        const mensaje = customMessage || document.getElementById('chatInput').value.trim();
        if (!mensaje) return;

        try {
            const response = await fetch('/lexipro/modules/chat/api_web.php', {
                method: 'POST',
                headers: { 'Content-Type': 'application/json' },
                body: JSON.stringify({
                    sesion_id: getSessionId(),
                    nombre: userData.nombre,
                    apellido: userData.apellido,
                    email: userData.email,
                    mensaje: mensaje,
                    es_cliente: 1
                })
            });

            const result = await response.json();

            if (result.success) {
                if (!isAuto) {
                    document.getElementById('chatInput').value = '';
                }
                loadMessages();
            }
        } catch (error) {
            console.error('Error al enviar mensaje:', error);
        }
    }

    async function loadMessages() {
        try {
            const response = await fetch(`/lexipro/modules/chat/api_web.php?sesion_id=${getSessionId()}`);
            const result = await response.json();

            if (result.success) {
                renderMessages(result.mensajes);
            }
        } catch (error) {
            console.error('Error al cargar mensajes:', error);
        }
    }

    function renderMessages(mensajes) {
        const container = document.getElementById('chatMessages');
        container.innerHTML = '';

        mensajes.forEach(msg => {
            const messageDiv = document.createElement('div');
            messageDiv.className = `chat-message ${msg.es_cliente == 1 ? 'user' : 'agent'}`;

            const bubble = document.createElement('div');
            bubble.className = 'message-bubble';
            bubble.textContent = msg.mensaje;

            const time = document.createElement('div');
            time.className = 'message-time';
            time.textContent = formatTime(msg.created_at);

            messageDiv.appendChild(bubble);
            messageDiv.appendChild(time);
            container.appendChild(messageDiv);
        });

        container.scrollTop = container.scrollHeight;
    }

    function formatTime(datetime) {
        const date = new Date(datetime);
        return date.toLocaleTimeString('es-PY', { hour: '2-digit', minute: '2-digit' });
    }

    function validateEmail(email) {
        return /^[^\s@]+@[^\s@]+\.[^\s@]+$/.test(email);
    }

    // Inicializar widget
    if (document.readyState === 'loading') {
        document.addEventListener('DOMContentLoaded', createChatWidget);
    } else {
        createChatWidget();
    }
})();
