document.addEventListener('DOMContentLoaded', function () {

    // --- ELEMENTOS DEL DOM ---
    const projectNameTitle = document.getElementById('project-name-title');
    const gridListContainer = document.getElementById('grid-list-container');
    
    const gridModalEl = document.getElementById('gridModal');
    const gridModal = new bootstrap.Modal(gridModalEl);
    const saveGridBtn = document.getElementById('save-grid-btn');
    const gridNetworksCheckboxes = document.getElementById('grid-networks-checkboxes');
    const gridProfessionalsCheckboxes = document.getElementById('grid-professionals-checkboxes');

    // --- URLs Y ESTADO ---
    const urlParams = new URLSearchParams(window.location.search);
    const projectId = urlParams.get('project_id');

    const PROJECTS_API_URL = 'api/projects.php';
    const GRIDS_API_URL = 'api/grids.php';
    const PROFESSIONALS_API_URL = 'api/professionals.php';

    const appState = {
        project: null,
        professionals: [],
        professionalsMap: new Map(),
        grids: [],
        isEditMode: false,
        currentGridId: null
    };

    // --- FUNCIONES DE CARGA INICIAL ---

    async function initialLoad() {
        if (!projectId) {
            projectNameTitle.textContent = "Proyecto no especificado";
            return;
        }

        try {
            const [projectRes, gridsRes, professionalsRes] = await Promise.all([
                fetch(`${PROJECTS_API_URL}?id=${projectId}`),
                fetch(`${GRIDS_API_URL}?project_id=${projectId}`),
                fetch(PROFESSIONALS_API_URL)
            ]);

            if (!projectRes.ok) throw new Error('No se pudo cargar el proyecto.');
            if (!gridsRes.ok) throw new Error('No se pudieron cargar las grillas.');
            if (!professionalsRes.ok) throw new Error('No se pudieron cargar los profesionales.');
            
            appState.project = await projectRes.json();
            appState.grids = await gridsRes.json();
            appState.professionals = await professionalsRes.json();
            appState.professionalsMap = new Map(appState.professionals.map(p => [p.id, p.nombre_completo]));

            projectNameTitle.textContent = `Grillas para: ${appState.project.nombre_proyecto}`;
            document.title = `Grillas: ${appState.project.nombre_proyecto} - Platflow`;

            populateNetworksForNewGrid();
            populateProfessionalsForNewGrid();
            renderGridList();

        } catch (error) {
            console.error("Error en la carga inicial:", error);
            projectNameTitle.textContent = "Error";
            gridListContainer.innerHTML = `<div class="alert alert-danger">${error.message}</div>`;
        }
    }

    function populateNetworksForNewGrid() {
        gridNetworksCheckboxes.innerHTML = ['Instagram', 'Facebook', 'LinkedIn', 'TikTok', 'X'].map(net => `
            <div class="form-check form-check-inline">
                <input class="form-check-input" type="checkbox" id="net-new-${net}" value="${net}">
                <label class="form-check-label" for="net-new-${net}">${net}</label>
            </div>
        `).join('');
    }

    function populateProfessionalsForNewGrid() {
        gridProfessionalsCheckboxes.innerHTML = (appState.professionals.filter(p => p.estado === 'Activo') || []).map(prof => `
            <div class="form-check">
                <input class="form-check-input" type="checkbox" id="prof-grid-${prof.id}" value="${prof.id}">
                <label class="form-check-label" for="prof-grid-${prof.id}">${prof.nombre_completo} <small class="text-muted">(${prof.puesto})</small></label>
            </div>
        `).join('');
    }

    async function fetchGrids() {
        try {
            const response = await fetch(`${GRIDS_API_URL}?project_id=${projectId}`);
            if (!response.ok) throw new Error('Error al recargar las grillas.');
            appState.grids = await response.json();
            renderGridList();
        } catch (error) {
            console.error("Error:", error);
            gridListContainer.innerHTML = `<div class="alert alert-danger">${error.message}</div>`;
        }
    }

    async function saveGrid() {
        const selectedNetworks = Array.from(gridNetworksCheckboxes.querySelectorAll('input:checked')).map(cb => cb.value);
        const selectedProfessionals = Array.from(gridProfessionalsCheckboxes.querySelectorAll('input:checked')).map(cb => parseInt(cb.value));

        const gridData = {
            project_id: projectId,
            name: document.getElementById('grid-name').value,
            start_date: document.getElementById('grid-start-date').value,
            end_date: document.getElementById('grid-end-date').value,
            networks: selectedNetworks,
            responsables: selectedProfessionals
        };

        if (!gridData.name || !gridData.start_date || !gridData.end_date) {
            alert('Nombre y fechas son obligatorios.');
            return;
        }

        const url = appState.isEditMode ? `${GRIDS_API_URL}?id=${appState.currentGridId}` : GRIDS_API_URL;
        const method = appState.isEditMode ? 'PUT' : 'POST';

        try {
            const response = await fetch(url, { method, headers: { 'Content-Type': 'application/json' }, body: JSON.stringify(gridData) });
            if (!response.ok) throw new Error(`Error al ${appState.isEditMode ? 'actualizar' : 'guardar'} la grilla.`);
            
            gridModal.hide();
            document.getElementById('grid-form').reset();
            fetchGrids();
        } catch (error) {
            console.error(error);
            alert(error.message);
        }
    }

    async function deleteGrid(gridId) {
        if (!confirm('¿Estás seguro de que quieres eliminar esta grilla y todos sus posteos? Esta acción no se puede deshacer.')) return;

        try {
            const response = await fetch(`${GRIDS_API_URL}?id=${gridId}`, { method: 'DELETE' });
            if (!response.ok) throw new Error('No se pudo eliminar la grilla.');
            fetchGrids();
        } catch (error) {
            console.error(error);
            alert(error.message);
        }
    }

    function handleGridEditClick(gridId) {
        const grid = appState.grids.find(g => g.id === gridId);
        if (!grid) return;

        appState.isEditMode = true;
        appState.currentGridId = gridId;

        document.getElementById('gridModalLabel').textContent = 'Editar Grilla de Contenido';
        document.getElementById('grid-name').value = grid.name;
        document.getElementById('grid-start-date').value = grid.start_date;
        document.getElementById('grid-end-date').value = grid.end_date;

        // Marcar checkboxes de redes
        gridNetworksCheckboxes.querySelectorAll('input').forEach(cb => {
            cb.checked = (grid.networks || []).includes(cb.value);
        });

        // Marcar checkboxes de responsables
        gridProfessionalsCheckboxes.querySelectorAll('input').forEach(cb => {
            cb.checked = (grid.responsables || []).includes(parseInt(cb.value));
        });

        gridModal.show();
    }

    function renderGridList() {
        gridListContainer.innerHTML = '';
        if (appState.grids.length === 0) {
            gridListContainer.innerHTML = '<p class="text-muted text-center col-12">No hay grillas creadas para este proyecto. ¡Crea la primera!</p>';
            return;
        }

        const formatDate = (dateString) => {
            if (!dateString) return 'N/A';
            const [year, month, day] = dateString.split('-');
            if (!year || !month || !day) return dateString;
            return `${day}/${month}/${year}`;
        };

        const getAssignedNames = (ids) => {
            if (!ids || ids.length === 0) return '<small class="text-muted">Sin asignar</small>';
            return ids.map(id => `<span class="badge bg-light text-dark me-1">${appState.professionalsMap.get(id) || 'N/A'}</span>`).join('');
        };

        appState.grids.forEach(grid => {
            const card = document.createElement('div');
            card.className = 'col-md-4 mb-4';
            card.innerHTML = `
                <div class="card h-100">
                    <div class="card-body d-flex flex-column">
                        <h5 class="card-title">${grid.name}</h5>
                        <p class="card-text text-muted small">
                            <i class="bi bi-calendar-range"></i> ${formatDate(grid.start_date)} al ${formatDate(grid.end_date)}
                        </p>
                        <div class="mb-2">
                            ${grid.networks.map(n => `<span class="badge bg-secondary me-1">${n}</span>`).join('')}
                        </div>
                        <div class="mb-3">
                            <small class="text-muted d-block">Responsables:</small>
                            ${getAssignedNames(grid.responsables)}
                        </div>
                        <div class="mt-auto d-flex justify-content-between">
                            <a href="grid-editor.html?grid_id=${grid.id}" class="btn btn-sm btn-primary">Abrir Editor</a>
                            <button class="btn btn-sm btn-outline-secondary edit-grid-btn" data-id="${grid.id}"><i class="bi bi-pencil"></i></button>
                            <button class="btn btn-sm btn-outline-danger delete-grid-btn" data-id="${grid.id}"><i class="bi bi-trash"></i></button>
                        </div>
                    </div>
                </div>
            `;
            gridListContainer.appendChild(card);
        });
    }

    // --- EVENT LISTENERS ---
    saveGridBtn.addEventListener('click', saveGrid);

    document.getElementById('add-grid-btn').addEventListener('click', () => {
        appState.isEditMode = false;
        appState.currentGridId = null;
        document.getElementById('gridModalLabel').textContent = 'Crear Nueva Grilla de Contenido';
        document.getElementById('grid-form').reset();
    });

    gridListContainer.addEventListener('click', (e) => {
        const deleteBtn = e.target.closest('.delete-grid-btn');
        if (deleteBtn) {
            deleteGrid(parseInt(deleteBtn.dataset.id));
        }

        const editBtn = e.target.closest('.edit-grid-btn');
        if (editBtn) {
            handleGridEditClick(parseInt(editBtn.dataset.id));
        }
    });

    // --- CARGA INICIAL ---
    initialLoad();
});