document.addEventListener('DOMContentLoaded', function () {

    // --- URLs DE API ---
    const SETTINGS_API_URL = 'api/settings.php';

    // --- ELEMENTOS DEL DOM ---
    const agencyForm = document.getElementById('agency-settings-form');
    const saveAgencyBtn = document.getElementById('save-settings-btn');
    const logoInput = document.getElementById('logo');
    const logoPreview = document.getElementById('logo-preview');

    const systemForm = document.getElementById('system-settings-form');
    const saveSystemBtn = document.getElementById('save-system-settings-btn');

    const aiForm = document.getElementById('ai-settings-form');
    const saveAiBtn = document.getElementById('save-ai-settings-btn');
    const aiProviderSelect = document.getElementById('ai_provider');

    // --- FUNCIONES ---

    /**
     * Carga los datos de configuración existentes desde la API y los muestra en el formulario.
     */
    async function loadSettings() {
        try {
            const response = await fetch(SETTINGS_API_URL);
            if (!response.ok) throw new Error('No se pudieron cargar los datos de configuración.');
            
            const settings = await response.json();
            const agencyData = settings.agency_data || {};
            const systemData = settings.system_settings || {};
            const aiData = settings.ai_settings || {};

            // Rellenar formulario de Agencia
            for (const key in agencyData) {
                const element = document.getElementById(key);
                if (element && key !== 'logo_url') {
                    element.value = agencyData[key];
                }
            }
            if (agencyData.logo_url) {
                logoPreview.src = agencyData.logo_url;
            }

            // Rellenar formulario de Ajustes del Sistema
            document.getElementById('thousands_separator').value = systemData.thousands_separator || '.';
            document.getElementById('decimal_separator').value = systemData.decimal_separator || ',';
            document.getElementById('decimal_places').value = systemData.decimal_places !== undefined ? systemData.decimal_places : 0;
            document.getElementById('percentage_decimals').value = systemData.percentage_decimals !== undefined ? systemData.percentage_decimals : 1;
            document.getElementById('currency_code').value = systemData.currency_code || 'PYG';
            document.getElementById('currency_symbol').value = systemData.currency_symbol || 'Gs.';
            
            // Rellenar formulario de IA
            const apiKeyField = document.getElementById('api_key');
            if (document.getElementById('ai_provider')) document.getElementById('ai_provider').value = aiData.ai_provider || 'gemini';
            if (document.getElementById('ai_model')) document.getElementById('ai_model').value = aiData.ai_model || '';
            // No mostramos la clave guardada, pero la mantenemos en un atributo para no perderla si no se cambia
            if (apiKeyField && aiData.api_key) apiKeyField.placeholder = "•••••••••••••••••••• (Clave guardada)";

        } catch (error) {
            console.error('Error al cargar la configuración:', error);
            // Podríamos mostrar una alerta al usuario aquí
        }
    }

    /**
     * Envía los datos del formulario (incluyendo el logo) a la API para guardarlos.
     */
    async function saveAgencySettings(event) {
        event.preventDefault(); // Prevenir el envío tradicional del formulario

        // Construir FormData manualmente para asegurar que todos los campos se incluyan
        const formData = new FormData();
        const formElements = agencyForm.elements;
        for (let i = 0; i < formElements.length; i++) {
            const element = formElements[i];
            if (element.name) {
                if (element.type === 'file') {
                    if (element.files.length > 0) formData.append(element.name, element.files[0]);
                } else {
                    formData.append(element.name, element.value);
                }
            }
        }

        saveAgencyBtn.disabled = true;
        saveAgencyBtn.innerHTML = '<span class="spinner-border spinner-border-sm" role="status" aria-hidden="true"></span> Guardando...';

        try {
            const response = await fetch(SETTINGS_API_URL, {
                method: 'POST',
                body: formData // FormData se encarga de los headers para multipart/form-data
            });

            if (!response.ok) throw new Error('Falló la solicitud para guardar la configuración.');

            const result = await response.json();
            alert(result.message || 'Configuración guardada.');
            
            // Actualizar la vista previa del logo si se subió uno nuevo
            if (result.data && result.data.agency_data && result.data.agency_data.logo_url) {
                logoPreview.src = result.data.agency_data.logo_url + '?t=' + new Date().getTime(); // Cache busting
            }

        } catch (error) {
            console.error('Error al guardar la configuración:', error);
            alert('Error: ' + error.message);
        } finally {
            saveAgencyBtn.disabled = false;
            saveAgencyBtn.innerHTML = 'Guardar Cambios';
        }
    }

    async function saveSystemSettings(event) {
        event.preventDefault();

        const formData = new FormData(systemForm);
        saveSystemBtn.disabled = true;
        saveSystemBtn.innerHTML = '<span class="spinner-border spinner-border-sm" role="status" aria-hidden="true"></span> Guardando...';

        try {
            const response = await fetch(SETTINGS_API_URL, {
                method: 'POST',
                body: formData
            });

            if (!response.ok) throw new Error('Falló la solicitud para guardar los ajustes.');

            const result = await response.json();
            alert(result.message || 'Ajustes guardados.');

        } catch (error) {
            console.error('Error al guardar los ajustes:', error);
            alert('Error: ' + error.message);
        } finally {
            saveSystemBtn.disabled = false;
            saveSystemBtn.innerHTML = 'Guardar Ajustes';
        }
    }

    async function saveAiSettings(event) {
        event.preventDefault();

        const formData = new FormData();
        formData.append('form_type', 'ai_settings');
        formData.append('ai_provider', document.getElementById('ai_provider').value);
        formData.append('ai_model', document.getElementById('ai_model').value);
        // Solo añadir la api_key al FormData si el usuario ha escrito algo nuevo.
        // Si no, el backend mantendrá la clave existente.
        if (document.getElementById('api_key').value) formData.append('api_key', document.getElementById('api_key').value);

        saveAiBtn.disabled = true;
        saveAiBtn.innerHTML = '<span class="spinner-border spinner-border-sm" role="status" aria-hidden="true"></span> Guardando...';

        try {
            const response = await fetch(SETTINGS_API_URL, {
                method: 'POST',
                body: formData
            });

            if (!response.ok) throw new Error('Falló la solicitud para guardar los ajustes de IA.');

            const result = await response.json();
            alert(result.message || 'Ajustes de IA guardados.');
            loadSettings(); // Recargamos los datos sin refrescar la página

        } catch (error) {
            console.error('Error al guardar los ajustes de IA:', error);
            alert('Error: ' + error.message);
        } finally {
            saveAiBtn.disabled = false;
            saveAiBtn.innerHTML = 'Guardar Configuración de IA';
        }
    }

    // --- EVENT LISTENERS ---
    if (agencyForm) {
        agencyForm.addEventListener('submit', saveAgencySettings);
    }
    if (systemForm) {
        systemForm.addEventListener('submit', saveSystemSettings);
    }
    if (aiForm) {
        aiForm.addEventListener('submit', saveAiSettings);
    }

    logoInput.addEventListener('change', () => {
        const file = logoInput.files[0];
        if (file) {
            logoPreview.src = URL.createObjectURL(file);
        }
    });

    // --- INICIO ---
    loadSettings();
});