document.addEventListener('DOMContentLoaded', function () {

    // --- ELEMENTOS DEL DOM ---
    const clientsTableBody = document.getElementById('clients-table-body');
    const saveClientBtn = document.getElementById('save-client-btn');
    const addClientForm = document.getElementById('add-client-form');
    const clientModalElement = document.getElementById('addClientModal');
    const addClientModal = new bootstrap.Modal(clientModalElement);
    const clientModalTitle = document.getElementById('addClientModalLabel');

    const contactsTableBody = document.getElementById('contacts-table-body');
    const saveContactBtn = document.getElementById('save-contact-btn');
    const addContactForm = document.getElementById('add-contact-form');
    const contactModalElement = document.getElementById('addContactModal');
    const addContactModal = new bootstrap.Modal(contactModalElement);
    const contactModalTitle = document.getElementById('addContactModalLabel');
    const empresaIdDropdown = document.getElementById('empresa_id');

    const viewClientModalElement = document.getElementById('viewClientModal');
    const viewClientModal = new bootstrap.Modal(viewClientModalElement);
    const viewClientModalLabel = document.getElementById('viewClientModalLabel');

    // --- URLs DE LA API ---
    const CLIENTS_API_URL = 'api/clients.php';
    const CONTACTS_API_URL = 'api/contacts.php';
    const PROJECTS_API_URL = 'api/projects.php';

    // --- ESTADO CENTRALIZADO ---
    const appState = {
        clients: [],
        contacts: [],
        projects: []
    };
    // --- ESTADO DE LA APLICACIÓN ---
    let clientEditMode = false;
    let currentClientId = null;
    let contactEditMode = false;
    let currentContactId = null;

    // --- FUNCIONES PARA CLIENTES ---

    async function fetchClients() {
        try {
            const response = await fetch(CLIENTS_API_URL);
            if (!response.ok) throw new Error('Error al obtener los clientes.');
            appState.clients = await response.json();
            renderClientsTable(appState.clients);
            populateClientsDropdown(appState.clients); // Popular el dropdown para contactos
        } catch (error) {
            console.error('Error:', error);
            clientsTableBody.innerHTML = `<tr><td colspan="7" class="text-center">${error.message}</td></tr>`;
        }
    }

    async function fetchSingleClient(id) {
        try {
            const response = await fetch(`${CLIENTS_API_URL}?id=${id}`);
            if (!response.ok) throw new Error('No se pudo encontrar al cliente.');
            return await response.json();
        } catch (error) {
            alert(error.message);
            console.error('Error:', error);
        }
    }

    async function saveClient() {
        const clientData = {
            nombre_empresa: document.getElementById('nombre_empresa').value,
            razon_social: document.getElementById('razon_social').value,
            ruc: document.getElementById('ruc').value,
            telefono: document.getElementById('telefono').value,
            correo: document.getElementById('correo').value,
            web: document.getElementById('web').value,
            direccion: document.getElementById('direccion').value,
            ciudad: document.getElementById('ciudad').value,
            pais: document.getElementById('pais').value,
            estado: document.getElementById('estado').value,
            link_fb: document.getElementById('link_fb').value,
            link_ig: document.getElementById('link_ig').value,
            link_in: document.getElementById('link_in').value,
            link_ytb: document.getElementById('link_ytb').value,
            link_x: document.getElementById('link_x').value,
            link_tiktok: document.getElementById('link_tiktok').value
        };

        const url = clientEditMode ? `${CLIENTS_API_URL}?id=${currentClientId}` : CLIENTS_API_URL;
        const method = clientEditMode ? 'PUT' : 'POST';

        try {
            const response = await fetch(url, { method, headers: { 'Content-Type': 'application/json' }, body: JSON.stringify(clientData) });
            if (!response.ok) throw new Error(`Error al ${clientEditMode ? 'actualizar' : 'crear'} el cliente.`);
            addClientModal.hide();
            fetchClients();
        } catch (error) {
            console.error('Error:', error); alert(error.message);
        }
    }

    async function deleteClient(id) {
        try {
            const response = await fetch(`${CLIENTS_API_URL}?id=${id}`, { method: 'DELETE' });
            if (!response.ok) throw new Error('Error al eliminar el cliente.');
            fetchClients();
        } catch (error) {
            console.error('Error:', error); alert(error.message);
        }
    }

    function renderClientsTable(clients) {
        clientsTableBody.innerHTML = '';
        if (clients.length === 0) {
            clientsTableBody.innerHTML = '<tr><td colspan="7" class="text-center">No hay clientes registrados.</td></tr>';
            return;
        }
        clients.forEach(client => {
            const row = document.createElement('tr');
            const statusBadge = client.estado === 'Activo' ? `<span class="badge bg-success">${client.estado}</span>` : `<span class="badge bg-secondary">${client.estado}</span>`;
            row.innerHTML = `
                <td>${client.id}</td>
                <td>${client.nombre_empresa}</td>
                <td>${client.ruc}</td>
                <td>${client.telefono}</td>
                <td>${client.correo}</td>
                <td>${statusBadge}</td>
                <td>
                    <button class="btn btn-sm btn-info view-client-btn" data-id="${client.id}" title="Ver Detalles"><i class="bi bi-eye-fill"></i></button>
                    <button class="btn btn-sm btn-outline-secondary edit-client-btn" data-id="${client.id}" title="Editar"><i class="bi bi-pencil-fill"></i></button>
                    <button class="btn btn-sm btn-outline-danger delete-client-btn" data-id="${client.id}" title="Eliminar"><i class="bi bi-trash-fill"></i></button>
                </td>
            `;
            clientsTableBody.appendChild(row);
        });
    }

    async function handleClientEditClick(id) {
        const client = await fetchSingleClient(id);
        if (!client) return;
        clientEditMode = true;
        currentClientId = id;
        clientModalTitle.textContent = 'Editar Cliente';
        try {
            Object.keys(client).forEach(key => {
                const el = document.getElementById(key);
                if (el) el.value = client[key];
            });
            addClientModal.show();
        } catch (error) {
            console.error("Error al poblar el formulario de cliente:", error);
            alert("No se pudo abrir el editor. Revisa la consola para más detalles.");
        }
    }

    function prepareAddClientModal() {
        clientEditMode = false;
        currentClientId = null;
        clientModalTitle.textContent = 'Añadir Nuevo Cliente';
        addClientForm.reset();
    }

    // --- FUNCIONES PARA CONTACTOS ---

    async function fetchContacts() {
        try {
            const response = await fetch(CONTACTS_API_URL);
            if (!response.ok) throw new Error('Error al obtener los contactos.');
            appState.contacts = await response.json();
            renderContactsTable(appState.contacts, appState.clients);
        } catch (error) { // El error de clientes ya se maneja en fetchClients
            console.error('Error:', error);
            contactsTableBody.innerHTML = `<tr><td colspan="7" class="text-center">${error.message}</td></tr>`;
        }
    }

    async function fetchProjects() {
        try {
            const response = await fetch(PROJECTS_API_URL);
            if (!response.ok) throw new Error('Error al obtener los proyectos.');
            appState.projects = await response.json();
        } catch (error) {
            console.error('Error:', error);
            // No mostramos error en la UI principal, ya que esta es una carga secundaria
        }
    }


    async function fetchSingleContact(id) {
        try {
            const response = await fetch(`${CONTACTS_API_URL}?id=${id}`);
            if (!response.ok) throw new Error('No se pudo encontrar al contacto.');
            return await response.json();
        } catch (error) {
            alert(error.message);
            console.error('Error:', error);
        }
    }

    async function saveContact() {
        const contactData = {
            empresa_id: document.getElementById('empresa_id').value,
            nombre_completo: document.getElementById('nombre_completo_contact').value,
            cargo: document.getElementById('cargo_contact').value,
            correo: document.getElementById('correo_contact').value,
            telefono: document.getElementById('telefono_contact').value
        };

        const url = contactEditMode ? `${CONTACTS_API_URL}?id=${currentContactId}` : CONTACTS_API_URL;
        const method = contactEditMode ? 'PUT' : 'POST';

        try {
            const response = await fetch(url, { method, headers: { 'Content-Type': 'application/json' }, body: JSON.stringify(contactData) });
            if (!response.ok) throw new Error(`Error al ${contactEditMode ? 'actualizar' : 'crear'} el contacto.`);
            addContactModal.hide();
            fetchContacts();
        } catch (error) {
            console.error('Error:', error); alert(error.message);
        }
    }

    async function deleteContact(id) {
        try {
            const response = await fetch(`${CONTACTS_API_URL}?id=${id}`, { method: 'DELETE' });
            if (!response.ok) throw new Error('Error al eliminar el contacto.');
            fetchContacts();
        } catch (error) {
            console.error('Error:', error); alert(error.message);
        }
    }

    function renderContactsTable(contacts, clients) {
        contactsTableBody.innerHTML = '';
        if (contacts.length === 0) {
            contactsTableBody.innerHTML = '<tr><td colspan="7" class="text-center">No hay contactos registrados.</td></tr>';
            return;
        }
        const clientMap = new Map(clients.map(c => [c.id, c.nombre_empresa]));
        contacts.forEach(contact => {
            const row = document.createElement('tr');
            row.innerHTML = `
                <td>${contact.id}</td>
                <td>${clientMap.get(parseInt(contact.empresa_id)) || 'N/A'}</td>
                <td>${contact.nombre_completo}</td>
                <td>${contact.cargo}</td>
                <td>${contact.correo}</td>
                <td>${contact.telefono}</td>
                <td>
                    <button class="btn btn-sm btn-outline-secondary edit-contact-btn" data-id="${contact.id}" title="Editar"><i class="bi bi-pencil-fill"></i></button>
                    <button class="btn btn-sm btn-outline-danger delete-contact-btn" data-id="${contact.id}" title="Eliminar"><i class="bi bi-trash-fill"></i></button>
                </td>
            `;
            contactsTableBody.appendChild(row);
        });
    }

    function populateClientsDropdown(clients) {
        empresaIdDropdown.innerHTML = '<option value="">Seleccione una empresa</option>';
        clients.forEach(client => {
            if (client.estado === 'Activo') {
                const option = document.createElement('option');
                option.value = client.id;
                option.textContent = client.nombre_empresa;
                empresaIdDropdown.appendChild(option);
            }
        });
    }

    async function handleContactEditClick(id) {
        const contact = await fetchSingleContact(id);
        if (!contact) return;
        contactEditMode = true;
        currentContactId = id;
        contactModalTitle.textContent = 'Editar Contacto';
        document.getElementById('empresa_id').value = contact.empresa_id;
        document.getElementById('nombre_completo_contact').value = contact.nombre_completo;
        document.getElementById('cargo_contact').value = contact.cargo;
        document.getElementById('correo_contact').value = contact.correo;
        document.getElementById('telefono_contact').value = contact.telefono;
        addContactModal.show();
    }

    function prepareAddContactModal() {
        contactEditMode = false;
        currentContactId = null;
        contactModalTitle.textContent = 'Añadir Nuevo Contacto';
        addContactForm.reset();
    }

    // --- FUNCIONES PARA VISTA DETALLADA ---

    async function handleClientViewClick(id) {
        const client = appState.clients.find(c => c.id === id);
        if (!client) return;

        // Filtrar contactos para este cliente
        const clientContacts = appState.contacts.filter(contact => parseInt(contact.empresa_id) === id);

        // Filtrar proyectos para este cliente
        const clientProjects = appState.projects.filter(project => parseInt(project.cliente_id) === id);

        // Poblar el modal con los datos
        viewClientModalLabel.textContent = `Detalles de: ${client.nombre_empresa}`;

        const clientDetailsView = document.getElementById('client-details-view');
        clientDetailsView.innerHTML = `
            <p><strong>Razón Social:</strong> ${client.razon_social || 'N/A'}</p>
            <p><strong>RUC:</strong> ${client.ruc || 'N/A'}</p>
            <p><strong>Teléfono:</strong> ${client.telefono || 'N/A'}</p>
            <p><strong>Correo:</strong> ${client.correo || 'N/A'}</p>
            <p><strong>Sitio Web:</strong> ${client.web ? `<a href="${client.web}" target="_blank">${client.web}</a>` : 'N/A'}</p>
            <p><strong>Dirección:</strong> ${client.direccion || 'N/A'}, ${client.ciudad || ''}, ${client.pais || ''}</p>
            <p><strong>Estado:</strong> ${client.estado}</p>
        `;

        const contactDetailsView = document.getElementById('contact-details-view');
        if (clientContacts.length > 0) {
            let contactsHtml = '<ul class="list-group">';
            clientContacts.forEach(contact => {
                contactsHtml += `
                    <li class="list-group-item">
                        <strong>${contact.nombre_completo}</strong> (${contact.cargo || 'Sin cargo'})<br>
                        <small class="text-muted">
                            <i class="bi bi-envelope-fill"></i> ${contact.correo || 'N/A'} | 
                            <i class="bi bi-telephone-fill"></i> ${contact.telefono || 'N/A'}
                        </small>
                    </li>
                `;
            });
            contactsHtml += '</ul>';
            contactDetailsView.innerHTML = contactsHtml;
        } else {
            contactDetailsView.innerHTML = '<p class="text-muted">No hay contactos registrados para este cliente.</p>';
        }

        const projectDetailsView = document.getElementById('project-details-view');
        if (clientProjects.length > 0) {
            let projectsHtml = '<ul class="list-group">';
            clientProjects.forEach(project => {
                projectsHtml += `
                    <li class="list-group-item d-flex justify-content-between align-items-center">
                        <a href="projects.html?search=${encodeURIComponent(project.nombre_proyecto)}">${project.nombre_proyecto}</a>
                        <span class="badge bg-info text-dark">${project.estado}</span>
                    </li>
                `;
            });
            projectsHtml += '</ul>';
            projectDetailsView.innerHTML = projectsHtml;
        } else {
            projectDetailsView.innerHTML = '<p class="text-muted">No hay proyectos asociados a este cliente.</p>';
        }


        // Mostrar el modal
        viewClientModal.show();
    }


    // --- EVENT LISTENERS ---

    clientsTableBody.addEventListener('click', function(event) {
        const button = event.target.closest('.view-client-btn, .edit-client-btn, .delete-client-btn');
        if (!button) return;

        const id = parseInt(button.dataset.id);

        if (button.classList.contains('delete-client-btn')) {
            if (confirm(`¿Estás seguro de que quieres eliminar al cliente con ID ${id}?`)) deleteClient(id);
        } else if (button.classList.contains('edit-client-btn')) {
            handleClientEditClick(id);
        } else if (button.classList.contains('view-client-btn')) {
            handleClientViewClick(id);
        }
    });

    contactsTableBody.addEventListener('click', function(event) {
        const button = event.target.closest('.edit-contact-btn, .delete-contact-btn');
        if (!button) return;

        const id = parseInt(button.dataset.id);

        if (button.classList.contains('delete-contact-btn')) {
            if (confirm(`¿Estás seguro de que quieres eliminar al contacto con ID ${id}?`)) deleteContact(id);
        } else if (button.classList.contains('edit-contact-btn')) {
            handleContactEditClick(id);
        }
    });

    saveClientBtn.addEventListener('click', saveClient);
    saveContactBtn.addEventListener('click', saveContact);

    document.querySelector('[data-bs-target="#addClientModal"]').addEventListener('click', prepareAddClientModal);
    document.querySelector('[data-bs-target="#addContactModal"]').addEventListener('click', prepareAddContactModal);

    clientModalElement.addEventListener('hidden.bs.modal', prepareAddClientModal);
    contactModalElement.addEventListener('hidden.bs.modal', prepareAddContactModal);

    // --- CARGA INICIAL ---
    fetchClients();
    fetchContacts();
    fetchProjects();
});