document.addEventListener('DOMContentLoaded', function () {

    // --- ELEMENTOS DEL DOM ---
    const tableBody = document.getElementById('all-tasks-table-body');
    const searchInput = document.getElementById('search-input');
    const projectFilter = document.getElementById('project-filter');
    const professionalFilter = document.getElementById('professional-filter');
    const priorityFilter = document.getElementById('priority-filter');
    const dateTypeFilter = document.getElementById('date-type-filter');
    const startDateFilter = document.getElementById('start-date-filter');
    const endDateFilter = document.getElementById('end-date-filter');
    const clearDatesBtn = document.getElementById('clear-dates-btn');

    // --- URLs DE LA API ---
    const TASKS_API_URL = 'api/tasks.php';
    const PROJECTS_API_URL = 'api/projects.php';
    const PROFESSIONALS_API_URL = 'api/professionals.php';

    // --- ESTADO ---
    let appState = {
        tasks: [],
        projects: [],
        professionals: [],
        sortColumn: 'fecha_fin',
        sortDirection: 'asc'
    };

    // --- FUNCIONES DE FORMATO ---
    const formatDate = (dateString) => {
        if (!dateString) return 'N/A';
        const [year, month, day] = dateString.split('-');
        if (!year || !month || !day) return dateString;
        return `${day}/${month}/${year}`;
    };

    // --- INICIALIZACIÓN ---
    async function initialLoad() {
        try {
            const [tasksRes, projectsRes, professionalsRes] = await Promise.all([
                fetch(TASKS_API_URL),
                fetch(PROJECTS_API_URL),
                fetch(PROFESSIONALS_API_URL)
            ]);

            appState.tasks = await tasksRes.json();
            appState.projects = await projectsRes.json();
            appState.professionals = await professionalsRes.json();

            populateFilters();
            renderTable();

        } catch (error) {
            console.error("Error en la carga inicial:", error);
            tableBody.innerHTML = `<tr><td colspan="7" class="text-center text-danger">Error al cargar los datos.</td></tr>`;
        }
    }

    // --- RENDERIZADO Y FILTROS ---
    function populateFilters() {
        // Proyectos
        projectFilter.innerHTML = '<option value="all">Todos los proyectos</option>';
        appState.projects.forEach(p => {
            projectFilter.innerHTML += `<option value="${p.id}">${p.nombre_proyecto}</option>`;
        });

        // Profesionales
        professionalFilter.innerHTML = '<option value="all">Todos los profesionales</option>';
        appState.professionals.forEach(p => {
            professionalFilter.innerHTML += `<option value="${p.id}">${p.nombre_completo}</option>`;
        });
    }

    function renderTable() {
        tableBody.innerHTML = '';
        const filteredAndSortedTasks = getFilteredAndSortedTasks();

        if (filteredAndSortedTasks.length === 0) {
            tableBody.innerHTML = '<tr><td colspan="7" class="text-center">No se encontraron tareas con los filtros aplicados.</td></tr>';
            return;
        }

        const projectMap = new Map(appState.projects.map(p => [p.id, p.nombre_proyecto]));
        const professionalMap = new Map(appState.professionals.map(p => [p.id, p.nombre_completo]));

        filteredAndSortedTasks.forEach(task => {
            const projectName = projectMap.get(parseInt(task.project_id)) || 'N/A';
            const professionalName = professionalMap.get(parseInt(task.asignado_a)) || 'N/A';

            let priorityBadge = '';
            switch (task.prioridad) {
                case 'Alta': priorityBadge = '<span class="badge bg-danger">Alta</span>'; break;
                case 'Baja': priorityBadge = '<span class="badge bg-secondary">Baja</span>'; break;
                default: priorityBadge = '<span class="badge bg-primary">Media</span>'; break;
            }

            const row = `
                <tr>
                    <td><a href="project-detail.html?id=${task.project_id}">${task.nombre_tarea}</a></td>
                    <td>${projectName}</td>
                    <td>${professionalName}</td>
                    <td>${priorityBadge}</td>
                    <td>${formatDate(task.fecha_fin)}</td>
                    <td>${formatDate(task.fecha_fin_real)}</td>
                    <td>
                        <div class="progress">
                            <div class="progress-bar" role="progressbar" style="width: ${task.porcentaje}%;" aria-valuenow="${task.porcentaje}" aria-valuemin="0" aria-valuemax="100">${task.porcentaje}%</div>
                        </div>
                    </td>
                </tr>
            `;
            tableBody.innerHTML += row;
        });
    }

    function getFilteredAndSortedTasks() {
        const searchTerm = searchInput.value.toLowerCase();
        const selectedProject = projectFilter.value;
        const selectedProfessional = professionalFilter.value;
        const selectedPriority = priorityFilter.value;
        const selectedDateType = dateTypeFilter.value;
        const startDate = startDateFilter.value;
        const endDate = endDateFilter.value;

        // 1. Filtrar
        let filteredTasks = appState.tasks.filter(task => {
            const projectMap = new Map(appState.projects.map(p => [p.id, p.nombre_proyecto]));
            const professionalMap = new Map(appState.professionals.map(p => [p.id, p.nombre_completo]));
            task.project_name = projectMap.get(parseInt(task.project_id)) || '';
            task.professional_name = professionalMap.get(parseInt(task.asignado_a)) || '';

            const searchMatch = task.nombre_tarea.toLowerCase().includes(searchTerm);
            const projectMatch = selectedProject === 'all' || task.project_id == selectedProject;
            const professionalMatch = selectedProfessional === 'all' || task.asignado_a == selectedProfessional;
            const priorityMatch = selectedPriority === 'all' || (task.prioridad || 'Media') === selectedPriority;

            let dateMatch = true;
            if (startDate && endDate) {
                const dateField = selectedDateType === 'real' ? task.fecha_fin_real : task.fecha_fin;
                if (dateField) {
                    dateMatch = dateField >= startDate && dateField <= endDate;
                } else {
                    dateMatch = false;
                }
            }

            return searchMatch && projectMatch && professionalMatch && priorityMatch && dateMatch;
        });

        // 2. Ordenar
        filteredTasks.sort((a, b) => {
            let valA = a[appState.sortColumn];
            let valB = b[appState.sortColumn];

            // Manejo para valores nulos o indefinidos
            if (valA == null) valA = '';
            if (valB == null) valB = '';

            let comparison = 0;
            if (valA > valB) {
                comparison = 1;
            } else if (valA < valB) {
                comparison = -1;
            }

            return appState.sortDirection === 'asc' ? comparison : -comparison;
        });

        return filteredTasks;
    }

    function handleSort(e) {
        const newSortColumn = e.target.dataset.sort;
        if (!newSortColumn) return;

        if (appState.sortColumn === newSortColumn) {
            appState.sortDirection = appState.sortDirection === 'asc' ? 'desc' : 'asc';
        } else {
            appState.sortColumn = newSortColumn;
            appState.sortDirection = 'asc';
        }

        // Actualizar indicadores visuales
        document.querySelectorAll('th[data-sort]').forEach(th => {
            th.classList.remove('sort-asc', 'sort-desc');
            if (th.dataset.sort === appState.sortColumn) {
                th.classList.add(appState.sortDirection === 'asc' ? 'sort-asc' : 'sort-desc');
            }
        });

        renderTable();
    }

    // --- LISTENERS ---
    [searchInput, projectFilter, professionalFilter, priorityFilter, dateTypeFilter, startDateFilter, endDateFilter].forEach(el => {
        el.addEventListener('input', renderTable);
    });

    clearDatesBtn.addEventListener('click', () => {
        startDateFilter.value = '';
        endDateFilter.value = '';
        renderTable();
    });

    document.querySelector('thead').addEventListener('click', handleSort);

    // --- INICIO ---
    initialLoad();
});