document.addEventListener('DOMContentLoaded', function() {

    // --- ELEMENTOS DEL DOM ---
    const calendarEl = document.getElementById('task-calendar');
    const professionalFilterEl = document.getElementById('professional-filter');
    const priorityFilterEl = document.getElementById('priority-filter');
    const tagFilterEl = document.getElementById('tag-filter');

    // --- URLs DE LA API ---
    const TASKS_API_URL = 'api/tasks.php';
    const PROFESSIONALS_API_URL = 'api/professionals.php';

    // --- ESTADO ---
    let appState = {
        tasks: [],
        professionals: [],
        calendar: null
    };

    // --- INICIALIZACIÓN ---
    async function initialLoad() {
        try {
            const [tasksRes, professionalsRes] = await Promise.all([
                fetch(TASKS_API_URL),
                fetch(PROFESSIONALS_API_URL)
            ]);

            appState.tasks = await tasksRes.json();
            appState.professionals = await professionalsRes.json();

            populateFilters();
            initializeCalendar();
            
        } catch (error) {
            console.error("Error en la carga inicial de la agenda:", error);
            calendarEl.innerHTML = `<div class="alert alert-danger">No se pudo cargar la agenda. ${error.message}</div>`;
        }
    }

    function initializeCalendar() {
        appState.calendar = new FullCalendar.Calendar(calendarEl, {
            initialView: 'dayGridMonth',
            locale: 'es',
            headerToolbar: {
                left: 'prev,next today',
                center: 'title',
                right: 'dayGridMonth,timeGridWeek,listWeek'
            },
            events: getFilteredEvents(),
            eventClick: function(info) {
                // Redirigir al detalle del proyecto de la tarea
                if (info.event.extendedProps.projectId) {
                    window.location.href = `project-detail.html?id=${info.event.extendedProps.projectId}`;
                }
            },
            eventDidMount: function(info) {
                // Añadir popover con más detalles
                new bootstrap.Popover(info.el, {
                    title: info.event.title,
                    content: info.event.extendedProps.description || 'Sin descripción.',
                    trigger: 'hover',
                    placement: 'top',
                    container: 'body',
                    html: true
                });
            }
        });
        appState.calendar.render();
    }

    // --- FILTROS Y EVENTOS ---

    function populateFilters() {
        // Profesionales
        professionalFilterEl.innerHTML = '<option value="all">Todos los Responsables</option>';
        appState.professionals.forEach(p => {
            professionalFilterEl.innerHTML += `<option value="${p.id}">${p.nombre_completo}</option>`;
        });

        // Prioridades
        priorityFilterEl.innerHTML = '<option value="all">Todas las Prioridades</option>';
        ['Alta', 'Media', 'Baja'].forEach(p => {
            priorityFilterEl.innerHTML += `<option value="${p}">${p}</option>`;
        });

        // Etiquetas (Tags)
        const allTags = new Set(appState.tasks.flatMap(t => t.tags || []));
        tagFilterEl.innerHTML = '<option value="all">Todas las Etiquetas</option>';
        allTags.forEach(tag => {
            tagFilterEl.innerHTML += `<option value="${tag}">${tag}</option>`;
        });
    }

    function getFilteredEvents() {
        const selectedProfessional = professionalFilterEl.value;
        const selectedPriority = priorityFilterEl.value;
        const selectedTag = tagFilterEl.value;

        const filteredTasks = appState.tasks.filter(task => {
            const professionalMatch = selectedProfessional === 'all' || task.asignado_a == selectedProfessional;
            const priorityMatch = selectedPriority === 'all' || (task.prioridad || 'Media') === selectedPriority;
            const tagMatch = selectedTag === 'all' || (task.tags && task.tags.includes(selectedTag));
            return professionalMatch && priorityMatch && tagMatch;
        });

        const events = [];
        filteredTasks.forEach(task => {
            // Evento para la fecha de fin estimada
            if (task.fecha_fin) {
                let color = '#0d6efd'; // Azul (Pendiente)
                let title = `Vence: ${task.nombre_tarea}`;
                if (task.porcentaje == 100) {
                    color = '#198754'; // Verde (Completada)
                } else if (new Date(task.fecha_fin) < new Date()) {
                    color = '#dc3545'; // Rojo (Vencida)
                }
                events.push({
                    title: title,
                    start: task.fecha_fin,
                    allDay: true,
                    backgroundColor: color,
                    borderColor: color,
                    extendedProps: { projectId: task.project_id, description: `Prioridad: ${task.prioridad || 'Media'}` }
                });
            }
            // Evento para la fecha de fin real
            if (task.fecha_fin_real) {
                events.push({
                    title: `Completada: ${task.nombre_tarea}`,
                    start: task.fecha_fin_real,
                    allDay: true,
                    backgroundColor: '#198754', // Verde
                    borderColor: '#198754',
                    extendedProps: { projectId: task.project_id, description: `Completada en fecha: ${task.fecha_fin_real}` }
                });
            }
        });
        return events;
    }

    function handleFilterChange() {
        appState.calendar.removeAllEvents();
        appState.calendar.addEventSource(getFilteredEvents());
    }

    // --- LISTENERS ---
    [professionalFilterEl, priorityFilterEl, tagFilterEl].forEach(el => el.addEventListener('change', handleFilterChange));

    // --- INICIO ---
    initialLoad();
});