<?php
header("Content-Type: application/json");

$database_path = '../config/database.json';

function get_database() {
    global $database_path;
    if (!file_exists($database_path)) {
        return ["professionals" => [], "clients" => [], "projects" => [], "contacts" => [], "tasks" => []];
    }
    $data = json_decode(file_get_contents($database_path), true);
    if (!isset($data['tasks'])) {
        $data['tasks'] = [];
    }
    return $data;
}

function save_database($data) {
    global $database_path;
    file_put_contents($database_path, json_encode($data, JSON_PRETTY_PRINT | JSON_UNESCAPED_UNICODE));
}

$method = $_SERVER['REQUEST_METHOD'];
$project_id = isset($_GET['project_id']) ? (int)$_GET['project_id'] : null;
$task_id = isset($_GET['id']) ? (int)$_GET['id'] : null;

if ($method === 'GET') {
    $db = get_database();
    $tasks = $db['tasks'] ?? [];

    if ($project_id !== null) {
        $project_tasks = array_values(array_filter($tasks, function($task) use ($project_id) {
            return isset($task['project_id']) && (int)$task['project_id'] === $project_id;
        }));
        echo json_encode($project_tasks);
    } else {
        echo json_encode($tasks);
    }
} elseif ($method === 'POST' && isset($_GET['action']) && $_GET['action'] === 'add_comment' && $task_id !== null) {
    // Endpoint para añadir un comentario a una tarea
    $input = json_decode(file_get_contents('php://input'), true);

    if (empty($input['text']) || !isset($input['author_id'])) {
        http_response_code(400);
        echo json_encode(['error' => 'El texto y el autor del comentario son requeridos.']);
        exit;
    }

    $db = get_database();
    $task_found = false;

    foreach ($db['tasks'] as $key => &$task) {
        if ($task['id'] === $task_id) {
            $new_comment = [
                'id' => uniqid('c_'), // ID único para el comentario
                'author_id' => $input['author_id'],
                'author_name' => $input['author_name'], // Asumimos que el frontend lo envía
                'text' => htmlspecialchars($input['text']), // Sanitizar el texto
                'timestamp' => date('Y-m-d H:i:s')
            ];
            if (!isset($task['comentarios'])) {
                $task['comentarios'] = [];
            }
            $task['comentarios'][] = $new_comment;
            $task_found = true;
            break;
        }
    }

    if ($task_found) {
        save_database($db);
        echo json_encode($new_comment);
    } else {
        http_response_code(404);
        echo json_encode(['error' => 'Tarea no encontrada']);
    }

} elseif ($method === 'POST') {
    $input = json_decode(file_get_contents('php://input'), true);

    if (empty($input['nombre_tarea']) || empty($input['project_id'])) {
        http_response_code(400);
        echo json_encode(['error' => 'El nombre de la tarea y el ID del proyecto son requeridos.']);
        exit;
    }

    $db = get_database();
    
    $new_id = 1;
    if (!empty($db['tasks'])) {
        $ids = array_column($db['tasks'], 'id');
        if (count($ids) > 0) {
            $new_id = max($ids) + 1;
        }
    }
    
    $new_task = [
        'id' => $new_id,
        'project_id' => (int)$input['project_id'],
        'nombre_tarea' => $input['nombre_tarea'],
        'asignado_a' => isset($input['asignado_a']) ? (int)$input['asignado_a'] : null,
        'fecha_inicio' => $input['fecha_inicio'] ?? null,
        'fecha_fin' => $input['fecha_fin'] ?? null,
        'fecha_fin_real' => null, // Nuevo campo, se inicializa en null
        'porcentaje' => isset($input['porcentaje']) ? (int)$input['porcentaje'] : 0,
        'prioridad' => $input['prioridad'] ?? 'Media',
        'comentarios' => [],
        'archivos' => [],
        'links' => [],
        'tags' => []
    ];

    $db['tasks'][] = $new_task;
    save_database($db);

    http_response_code(201);
    echo json_encode($new_task);

} elseif ($method === 'PUT' && $task_id !== null) {
    $input = json_decode(file_get_contents('php://input'), true);
    $db = get_database();
    
    $updated_task = null;
    $found = false;

    foreach ($db['tasks'] as $key => &$task) {
        if ($task['id'] === $task_id) {
            // Lógica específica para añadir archivos o tags sin sobreescribir todo
            if (isset($input['new_file'])) {
                if (!isset($task['archivos'])) $task['archivos'] = [];
                $task['archivos'][] = $input['new_file'];
            }
            if (isset($input['new_tag'])) {
                if (!isset($task['tags'])) $task['tags'] = [];
                if (!in_array($input['new_tag'], $task['tags'])) {
                    $task['tags'][] = $input['new_tag'];
                }
            }
            if (isset($input['new_link'])) {
                if (!isset($task['links'])) $task['links'] = [];
                $task['links'][] = $input['new_link'];
            }
            // Fusionar otros cambios generales
            $task = array_merge($task, $input);
            $updated_task = $task;
            $found = true;
            break;
        }
    }

    if ($found) {
        save_database($db);
        echo json_encode($updated_task);
    } else {
        http_response_code(404);
        echo json_encode(['error' => 'Tarea no encontrada para actualizar']);
    }

} elseif ($method === 'DELETE' && $task_id !== null) {
    $db = get_database();
    
    $original_count = count($db['tasks']);
    $db['tasks'] = array_values(array_filter($db['tasks'], function($task) use ($task_id) {
        return $task['id'] !== $task_id;
    }));
    
    if (count($db['tasks']) < $original_count) {
        save_database($db);
        http_response_code(204);
    } else {
        http_response_code(404);
        echo json_encode(['error' => 'Tarea no encontrada']);
    }

} else {
    http_response_code(405);
    echo json_encode(['error' => 'Método no permitido']);
}
?>