<?php
// Backend para gestionar los profesionales (CRUD)

header("Content-Type: application/json");

$database_path = '../config/database.json';

/**
 * Lee y decodifica el archivo JSON de la base de datos.
 * @return array
 */
function get_database() {
    global $database_path;
    if (!file_exists($database_path)) {        
        return ["professionals" => [], "clients" => [], "projects" => []];
    }
    $json_data = file_get_contents($database_path);
    return json_decode($json_data, true);
}

// Función para guardar la base de datos
function save_database($data) {
    global $database_path;
    $json_data = json_encode($data, JSON_PRETTY_PRINT | JSON_UNESCAPED_UNICODE);
    file_put_contents($database_path, $json_data);
}

/**
 * Maneja la subida de un archivo de imagen.
 * @param array $file - El array del archivo de $_FILES.
 * @return string|null - La URL del archivo guardado o null si hay error.
 */
function handle_file_upload($file) {
    if (isset($file) && $file['error'] === UPLOAD_ERR_OK) {
        $upload_dir = '../uploads/professionals/';
        if (!is_dir($upload_dir)) {
            mkdir($upload_dir, 0777, true);
        }
        $file_extension = pathinfo($file['name'], PATHINFO_EXTENSION);
        $file_name = 'prof_' . uniqid() . '.' . $file_extension;
        $target_path = $upload_dir . $file_name;

        if (move_uploaded_file($file['tmp_name'], $target_path)) {
            // Devuelve la URL relativa accesible desde el cliente
            return 'uploads/professionals/' . $file_name;
        }
    }
    return null;
}

$method = $_SERVER['REQUEST_METHOD'];
$id = isset($_GET['id']) ? (int)$_GET['id'] : null;
$id = $id ?: (isset($_POST['id']) ? (int)$_POST['id'] : null); // También buscar ID en POST para edición

if ($method === 'GET') {
    $db = get_database();
    if ($id !== null) {
        $professional = array_values(array_filter($db['professionals'], function($prof) use ($id) {
            return $prof['id'] === $id;
        }));
        if (count($professional) > 0) {
            echo json_encode($professional[0]);
        } else {
            http_response_code(404);
            echo json_encode(['error' => 'Profesional no encontrado']);
        }
    } else {
        echo json_encode($db['professionals']);
    }
} elseif ($method === 'POST') {
    $db = get_database();
    $input = $_POST;

    if ($id) { // --- MODO EDICIÓN ---
        $found_key = null;
        foreach ($db['professionals'] as $key => $prof) {
            if ($prof['id'] === $id) {
                $found_key = $key;
                break;
            }
        }

        if ($found_key !== null) {
            $professional = &$db['professionals'][$found_key];
            $professional['nombre_completo'] = $input['nombre_completo'] ?? $professional['nombre_completo'];
            $professional['puesto'] = $input['puesto'] ?? $professional['puesto'];
            $professional['tipo_contrato'] = $input['tipo_contrato'] ?? $professional['tipo_contrato'];
            $professional['fecha_ingreso'] = $input['fecha_ingreso'] ?? $professional['fecha_ingreso'];
            $professional['salario_mensual'] = $input['salario_mensual'] ?? $professional['salario_mensual'];
            $professional['email'] = $input['email'] ?? $professional['email'];
            $professional['telefono'] = $input['telefono'] ?? $professional['telefono'];
            $professional['estado'] = $input['estado'] ?? $professional['estado'];

            // Actualizar contraseña solo si se proporciona una nueva
            if (!empty($input['password'])) {
                $professional['password'] = password_hash($input['password'], PASSWORD_DEFAULT);
            }

            // Manejar subida de foto
            if (isset($_FILES['foto'])) {
                $new_foto_url = handle_file_upload($_FILES['foto']);
                if ($new_foto_url) {
                    // Opcional: borrar la foto antigua si existe
                    if (!empty($professional['foto_url']) && file_exists('../' . $professional['foto_url'])) {
                        unlink('../' . $professional['foto_url']);
                    }
                    $professional['foto_url'] = $new_foto_url;
                }
            }
            
            save_database($db);
            echo json_encode(['success' => true, 'professional' => $professional]);
        } else {
            http_response_code(404);
            echo json_encode(['success' => false, 'error' => 'Profesional no encontrado para actualizar']);
        }
    } else { // --- MODO CREACIÓN ---
        $new_id = empty($db['professionals']) ? 1 : max(array_column($db['professionals'], 'id')) + 1;
        
        $new_professional = [
            'id' => $new_id,
            'nombre_completo' => $input['nombre_completo'] ?? '',
            'puesto' => $input['puesto'] ?? '',
            'tipo_contrato' => $input['tipo_contrato'] ?? '',
            'fecha_ingreso' => $input['fecha_ingreso'] ?? '',
            'salario_mensual' => $input['salario_mensual'] ?? 0,
            'email' => $input['email'] ?? '',
            'telefono' => $input['telefono'] ?? '',
            'estado' => $input['estado'] ?? 'Inactivo',
            'password' => password_hash($input['password'] ?? 'password123', PASSWORD_DEFAULT),
            'foto_url' => handle_file_upload($_FILES['foto'] ?? null)
        ];

        $db['professionals'][] = $new_professional;
        save_database($db);

        http_response_code(201);
        echo json_encode(['success' => true, 'professional' => $new_professional]);
    }
} elseif ($method === 'DELETE' && $id !== null) {
    $db = get_database();
    
    // --- Lógica de prevención de borrado ---
    // Verificar si el profesional está asignado a algún proyecto
    $is_assigned = false;
    if (isset($db['projects']) && is_array($db['projects'])) {
        foreach ($db['projects'] as $project) {
            if (isset($project['profesionales_asignados']) && is_array($project['profesionales_asignados'])) {
                if (in_array($id, $project['profesionales_asignados'])) {
                    $is_assigned = true;
                    break; // Encontrado, no es necesario seguir buscando
                }
            }
        }
    }

    if ($is_assigned) {
        http_response_code(409); // 409 Conflict
        echo json_encode(['success' => false, 'error' => 'No se puede eliminar al profesional porque está asignado a uno o más proyectos.']);
        exit;
    }

    $original_count = count($db['professionals']);
    $db['professionals'] = array_filter($db['professionals'], function($prof) use ($id) {
        return $prof['id'] !== $id;
    });
    
    if (count($db['professionals']) < $original_count) {
        $db['professionals'] = array_values($db['professionals']);
        save_database($db);
        echo json_encode(['success' => true]);
    } else {
        http_response_code(404);
        echo json_encode(['success' => false, 'error' => 'Profesional no encontrado']);
    }

} else {
    http_response_code(405);
    echo json_encode(['error' => 'Método no permitido o ID no proporcionado']);
}

?>