<?php
// Backend para gestionar los clientes (CRUD)

header("Content-Type: application/json");

$database_path = '../config/database.json';

// Función para leer la base de datos
function get_database() {
    global $database_path;
    if (!file_exists($database_path)) {
        // Si no existe, lo creamos con la estructura base
        $base_structure = ["professionals" => [], "clients" => [], "projects" => []];
        file_put_contents($database_path, json_encode($base_structure, JSON_PRETTY_PRINT));
        return $base_structure;
    }
    $json_data = file_get_contents($database_path);
    return json_decode($json_data, true);
}

// Función para guardar la base de datos
function save_database($data) {
    global $database_path;
    $json_data = json_encode($data, JSON_PRETTY_PRINT | JSON_UNESCAPED_UNICODE);
    file_put_contents($database_path, $json_data);
}

$method = $_SERVER['REQUEST_METHOD'];
$id = isset($_GET['id']) ? (int)$_GET['id'] : null;

if ($method === 'GET') {
    $db = get_database();
    if ($id !== null) {
        $client = array_values(array_filter($db['clients'], function($cli) use ($id) {
            return $cli['id'] === $id;
        }));
        if (count($client) > 0) {
            echo json_encode($client[0]);
        } else {
            http_response_code(404);
            echo json_encode(['error' => 'Cliente no encontrado']);
        }
    } else {
        echo json_encode($db['clients']);
    }
} elseif ($method === 'POST') {
    $input = json_decode(file_get_contents('php://input'), true);

    $db = get_database();
    
    $new_id = 1;
    if (!empty($db['clients'])) {
        $ids = array_column($db['clients'], 'id');
        $new_id = max($ids) + 1;
    }
    
    $new_client = [
        'id' => $new_id,
        'nombre_empresa' => $input['nombre_empresa'] ?? '',
        'razon_social' => $input['razon_social'] ?? '',
        'ruc' => $input['ruc'] ?? '',
        'telefono' => $input['telefono'] ?? '',
        'correo' => $input['correo'] ?? '',
        'web' => $input['web'] ?? '',
        'direccion' => $input['direccion'] ?? '',
        'ciudad' => $input['ciudad'] ?? '',
        'pais' => $input['pais'] ?? '',
        'estado' => $input['estado'] ?? 'Inactivo',
        'link_fb' => $input['link_fb'] ?? '',
        'link_ig' => $input['link_ig'] ?? '',
        'link_in' => $input['link_in'] ?? '',
        'link_ytb' => $input['link_ytb'] ?? '',
        'link_x' => $input['link_x'] ?? '',
        'link_tiktok' => $input['link_tiktok'] ?? ''
    ];

    $db['clients'][] = $new_client;
    save_database($db);

    http_response_code(201);
    echo json_encode($new_client);

} elseif ($method === 'PUT' && $id !== null) {
    $input = json_decode(file_get_contents('php://input'), true);
    $db = get_database();
    
    $updated_client = null;
    $found = false;

    foreach ($db['clients'] as $key => $cli) {
        if ($cli['id'] === $id) {
            $db['clients'][$key] = array_merge($cli, $input);
            $db['clients'][$key]['id'] = $id;
            $updated_client = $db['clients'][$key];
            $found = true;
            break;
        }
    }

    if ($found) {
        save_database($db);
        echo json_encode($updated_client);
    } else {
        http_response_code(404);
        echo json_encode(['error' => 'Cliente no encontrado para actualizar']);
    }

} elseif ($method === 'DELETE' && $id !== null) {
    $db = get_database();
    
    // --- Lógica de prevención de borrado ---
    // Verificar si hay contactos asociados
    $associated_contacts = array_filter($db['contacts'] ?? [], function($contact) use ($id) {
        return isset($contact['empresa_id']) && (int)$contact['empresa_id'] === $id;
    });

    // Verificar si hay proyectos asociados
    $associated_projects = array_filter($db['projects'] ?? [], function($project) use ($id) {
        return isset($project['cliente_id']) && (int)$project['cliente_id'] === $id;
    });

    if (count($associated_contacts) > 0 || count($associated_projects) > 0) {
        http_response_code(409); // 409 Conflict es un buen código para esta situación
        echo json_encode(['error' => 'No se puede eliminar el cliente porque tiene contactos o proyectos asociados. Por favor, elimínelos o reasígnelos primero.']);
        exit;
    }

    $original_count = count($db['clients']);
    $db['clients'] = array_filter($db['clients'], function($cli) use ($id) {
        return $cli['id'] !== $id;
    });
    
    if (count($db['clients']) < $original_count) {
        $db['clients'] = array_values($db['clients']);
        save_database($db);
        http_response_code(204);
    } else {
        http_response_code(404);
        echo json_encode(['error' => 'Cliente no encontrado']);
    }

} else {
    http_response_code(405);
    echo json_encode(['error' => 'Método no permitido o ID no proporcionado']);
}

?>
