<?php
header("Content-Type: application/json");

function get_settings() {
    $database_path = '../config/database.json';
    if (!file_exists($database_path)) {
        return [];
    }
    $json_data = file_get_contents($database_path);
    $db = json_decode($json_data, true);
    return $db['settings'] ?? [];
}

$method = $_SERVER['REQUEST_METHOD'];

if ($method !== 'POST') {
    http_response_code(405);
    echo json_encode(['error' => 'Método no permitido']);
    exit;
}

$input = json_decode(file_get_contents('php://input'), true);
$task_type = $input['task_type'] ?? null;
$context = $input['context'] ?? [];

if (!$task_type) {
    http_response_code(400);
    echo json_encode(['error' => 'El tipo de tarea de IA es requerido.']);
    exit;
}

$settings = get_settings();
$ai_settings = $settings['ai_settings'] ?? null;

if (!$ai_settings || empty($ai_settings['api_key']) || empty($ai_settings['model'])) {
    http_response_code(409); // Conflict
    echo json_encode(['error' => 'La configuración de la IA no está completa. Por favor, configure el proveedor, la API Key y el modelo en la página de Configuración.']);
    exit;
}

// --- CONSTRUCCIÓN DE PROMPTS ---
$prompt = '';
switch ($task_type) {
    case 'generate_strategy':
        $prompt = "Actúa como un Director de Estrategia de Marketing. Para la empresa '{$context['clientName']}', y su proyecto '{$context['projectName']}', ".
                  "cuyos objetivos son: '{$context['projectDescription']}', y con un enfoque en marketing '{$context['marketingType']}', ".
                  "desarrolla una propuesta de marketing. Responde únicamente con un JSON que contenga tres claves: 'estrategico', 'tactico' y 'operativo'.";
        break;
    case 'generate_grid_posts':
        $prompt = "Actúa como un experto Community Manager. Basado en la siguiente estrategia de marketing:\n" .
                  "- Estratégico: {$context['estrategico']}\n- Táctico: {$context['tactico']}\n- Operativo: {$context['operativo']}\n" .
                  "Genera una lista de 3 a 5 ideas de posteos para redes sociales para los próximos 10 días. " .
                  "Responde únicamente con un array JSON de objetos. Cada objeto debe tener las siguientes claves: " .
                  "'fecha_posteo' (YYYY-MM-DD), 'hora_posteo' (HH:MM), 'tipo_contenido', 'redes' (array), 'objetivo', 'hook', 'copy', 'hashtags', 'cta', 'publico_objetivo' y 'resultado_esperado'.";
        break;
    case 'generate_creative_asset':
        $prompt = "Actúa como un Director Creativo. Basado en la siguiente información de un posteo:\n" .
                  "- Tipo: {$context['tipo_contenido']}\n- Objetivo: {$context['objetivo']}\n- Copy: {$context['copy']}\n" .
                  "Genera una descripción detallada para un activo creativo visual (prompt para DALL-E, Midjourney, etc.) o un guion corto para un video. " .
                  "Responde únicamente con un JSON que contenga una clave: 'creative_asset'.";
        break;
    default:
        http_response_code(400);
        echo json_encode(['error' => 'Tipo de tarea de IA no válido.']);
        exit;
}

// --- LLAMADA REAL A LA API DE IA (GEMINI) ---

$api_key = $ai_settings['api_key'];
$model = $ai_settings['model'];
$url = "https://generativelanguage.googleapis.com/v1beta/models/{$model}:generateContent?key={$api_key}";

$data = [
    'contents' => [
        [
            'parts' => [
                ['text' => $prompt]
            ]
        ]
    ]
];

$ch = curl_init($url);
curl_setopt($ch, CURLOPT_RETURNTRANSFER, true);
curl_setopt($ch, CURLOPT_POST, true);
curl_setopt($ch, CURLOPT_POSTFIELDS, json_encode($data));
curl_setopt($ch, CURLOPT_HTTPHEADER, [
    'Content-Type: application/json'
]);

$response = curl_exec($ch);
$http_code = curl_getinfo($ch, CURLINFO_HTTP_CODE);
curl_close($ch);

if ($http_code !== 200) {
    http_response_code($http_code);
    $error_response = json_decode($response, true);
    $error_message = $error_response['error']['message'] ?? 'Error desconocido al contactar la API de IA.';
    echo json_encode(['error' => "Error de la API de IA: " . $error_message]);
    exit;
}

$result = json_decode($response, true);

// Extraer el contenido de texto de la respuesta de Gemini
$text_content = $result['candidates'][0]['content']['parts'][0]['text'] ?? null;

if ($text_content === null) {
    http_response_code(500);
    echo json_encode(['error' => 'La IA devolvió una respuesta vacía o en un formato inesperado.']);
    exit;
}

// Como le pedimos a la IA que responda con un JSON, decodificamos ese texto.
$response_data = json_decode(trim($text_content), true); // Intento inicial

// Si el primer intento falla, podría ser porque la IA añadió texto extra.
// Intentamos extraer el JSON del texto.
if (json_last_error() !== JSON_ERROR_NONE) {
    $response_data = null; // Reseteamos por si el primer decode dio un resultado parcial
    preg_match('/\{.*\}|\[.*\]/s', $text_content, $matches);
    if (isset($matches[0])) {
        $response_data = json_decode($matches[0], true);
    }
}

if ($response_data === null) {
    http_response_code(500);
    echo json_encode(['error' => 'La IA devolvió una respuesta, pero no se pudo interpretar como un JSON válido.']);
    exit;
}

echo json_encode($response_data);
?>