<?php
// Establecer la cabecera para devolver JSON, importante para la comunicación con JavaScript
header('Content-Type: application/json');

// Ruta al archivo JSON donde se guardarán los mensajes.
// Usamos una ruta absoluta desde la raíz del servidor para mayor robustez.
// $_SERVER['DOCUMENT_ROOT'] apunta a 'c:/xampp/htdocs'
$json_file = $_SERVER['DOCUMENT_ROOT'] . '/webcopesa/data/messages.json';

// Función para enviar una respuesta de error estandarizada
function send_error($message) {
    http_response_code(400); // Bad Request
    echo json_encode(['success' => false, 'message' => $message]);
    exit; // Detener la ejecución
}

// Solo permitir solicitudes con el método POST
if ($_SERVER['REQUEST_METHOD'] !== 'POST') {
    send_error('Método no permitido. Se esperaba POST.');
}

// Leer el cuerpo de la solicitud, que viene en formato JSON
$data = json_decode(file_get_contents('php://input'), true);

// --- Verificación de reCAPTCHA v3 ---
$config_file_recaptcha = $_SERVER['DOCUMENT_ROOT'] . '/webcopesa/data/config.json';
if (file_exists($config_file_recaptcha)) {
    $config = json_decode(file_get_contents($config_file_recaptcha), true);
    $secret_key = $config['recaptcha_secret_key'] ?? '';

    if (!empty($secret_key)) {
        $token = $data['recaptcha_token'] ?? '';
        if (empty($token)) {
            send_error('Verificación de seguridad fallida. Token no encontrado.');
        }

        $verify_url = 'https://www.google.com/recaptcha/api/siteverify';
        $response = file_get_contents($verify_url . '?secret=' . $secret_key . '&response=' . $token);
        $response_data = json_decode($response, true);

        if (!$response_data['success'] || $response_data['score'] < 0.5) {
            send_error('Verificación de seguridad fallida. Eres un robot?');
        }
    }
}

// Validar que los datos necesarios estén presentes y no estén vacíos
$name = trim($data['name'] ?? '');
$email = trim($data['email'] ?? '');
$message = trim($data['message'] ?? '');

if (empty($name) || empty($email) || empty($message)) {
    send_error('Todos los campos son obligatorios.');
}

if (!filter_var($email, FILTER_VALIDATE_EMAIL)) {
    send_error('El formato del email no es válido.');
}

// Construir el nuevo mensaje con datos limpios y seguros
$new_message = [
    'id' => uniqid('msg_'), // Generar un ID único para cada mensaje
    'name' => htmlspecialchars($name),
    'email' => htmlspecialchars($email),
    'message' => htmlspecialchars($message),
    'date' => date('d/m/Y H:i:s'), // Añadir fecha y hora del envío
    'read' => false // Marcar como no leído por defecto
];

// Leer los mensajes existentes del archivo JSON
$messages = [];

// Verificación robusta: Asegurarse de que el directorio 'data' y el archivo 'messages.json' existan.
$data_dir = dirname($json_file);
if (!is_dir($data_dir)) {
    // Intentar crear el directorio 'data' si no existe.
    mkdir($data_dir, 0777, true);
}
if (!file_exists($json_file)) {
    file_put_contents($json_file, '[]');
}

if (file_exists($json_file)) {
    $json_content = file_get_contents($json_file);
    $messages = json_decode($json_content, true) ?: []; // Si el archivo está vacío o corrupto, empezar con un array vacío
}

// Añadir el nuevo mensaje al principio del array (para que los más nuevos aparezcan primero)
array_unshift($messages, $new_message);

// Guardar el array actualizado de vuelta en el archivo JSON
if (file_put_contents($json_file, json_encode($messages, JSON_PRETTY_PRINT | JSON_UNESCAPED_UNICODE))) {
    echo json_encode(['success' => true, 'message' => 'Mensaje enviado con éxito.']);
} else {
    http_response_code(500); // Internal Server Error
    echo json_encode(['success' => false, 'message' => 'Error al guardar el mensaje.']);
}