<?php
header('Content-Type: application/json');

$users_file = $_SERVER['DOCUMENT_ROOT'] . '/webcopesa/data/users.json';

function send_response($data, $code = 200) {
    http_response_code($code);
    echo json_encode($data);
    exit;
}

function get_users($file) {
    if (!file_exists($file)) {
        return [];
    }
    $json_content = file_get_contents($file);
    return json_decode($json_content, true) ?: [];
}

function save_users($file, $users) {
    return file_put_contents($file, json_encode($users, JSON_PRETTY_PRINT | JSON_UNESCAPED_UNICODE));
}

$method = $_SERVER['REQUEST_METHOD'];

if ($method === 'GET') {
    $users = get_users($users_file);
    // Por seguridad, nunca devolver las contraseñas hasheadas al cliente
    $safe_users = array_map(function($user) {
        unset($user['password']);
        return $user;
    }, $users);
    send_response($safe_users);
}

if ($method === 'POST') {
    $data = json_decode(file_get_contents('php://input'), true);
    $action = $data['action'] ?? 'create';

    if ($action === 'create') {
        $full_name = trim($data['full_name'] ?? '');
        $username = trim($data['username'] ?? '');
        $password = $data['password'] ?? '';

        if (empty($full_name) || empty($username) || empty($password)) {
            send_response(['success' => false, 'message' => 'Todos los campos son obligatorios.'], 400);
        }

        $users = get_users($users_file);

        // Verificar si el usuario ya existe
        foreach ($users as $user) {
            if (strtolower($user['username']) === strtolower($username)) {
                send_response(['success' => false, 'message' => 'El nombre de usuario ya está en uso.'], 400);
            }
        }

        $new_user = [
            'id' => uniqid('user_'),
            'full_name' => htmlspecialchars($full_name),
            'username' => htmlspecialchars($username),
            'password' => password_hash($password, PASSWORD_DEFAULT) // Hashear la contraseña
        ];

        array_push($users, $new_user);

        if (save_users($users_file, $users)) {
            send_response(['success' => true, 'message' => 'Usuario creado con éxito.']);
        } else {
            send_response(['success' => false, 'message' => 'Error al guardar el usuario.'], 500);
        }
    }

    if ($action === 'edit') {
        $user_id = $data['id'] ?? null;
        if (empty($user_id)) {
            send_response(['success' => false, 'message' => 'No se proporcionó el ID del usuario.'], 400);
        }

        $full_name = trim($data['full_name'] ?? '');
        $username = trim($data['username'] ?? '');
        $password = $data['password'] ?? '';

        // En modo edición, la contraseña es opcional.
        if (empty($full_name) || empty($username)) {
            send_response(['success' => false, 'message' => 'El nombre completo y el nombre de usuario son obligatorios.'], 400);
        }

        $users = get_users($users_file);

        // Verificar si el usuario ya existe
        $user_found = false;
        foreach ($users as $key => &$user) { // Usar referencia (&) para modificar el array directamente
            // Verificar unicidad del nombre de usuario, excluyendo al propio usuario
            if (strtolower($user['username']) === strtolower($username) && $user['id'] !== $user_id) {
                send_response(['success' => false, 'message' => 'El nombre de usuario ya está en uso.'], 400);
            }

            if ($user['id'] === $user_id) {
                $user['full_name'] = htmlspecialchars($full_name);
                $user['username'] = htmlspecialchars($username);
                // Solo actualizar la contraseña si se proporcionó una nueva
                if (!empty($password)) {
                    $user['password'] = password_hash($password, PASSWORD_DEFAULT);
                }
                $user_found = true;
            }
        }

        if (!$user_found) {
            send_response(['success' => false, 'message' => 'Usuario no encontrado.'], 404);
        }

        if (save_users($users_file, $users)) {
            send_response(['success' => true, 'message' => 'Usuario actualizado con éxito.']);
        } else {
            send_response(['success' => false, 'message' => 'Error al actualizar el usuario.'], 500);
        }
    }

    if ($action === 'delete') {
        $user_id = $data['id'] ?? null;
        if (empty($user_id)) {
            send_response(['success' => false, 'message' => 'No se proporcionó el ID del usuario.'], 400);
        }

        $users = get_users($users_file);

        // No permitir eliminar el único usuario
        if (count($users) <= 1) {
            send_response(['success' => false, 'message' => 'No se puede eliminar el único usuario administrador.'], 400);
        }

        $updated_users = array_filter($users, function($user) use ($user_id) {
            return $user['id'] !== $user_id;
        });

        $updated_users = array_values($updated_users);

        if (save_users($users_file, $updated_users)) {
            send_response(['success' => true, 'message' => 'Usuario eliminado con éxito.']);
        } else {
            send_response(['success' => false, 'message' => 'Error al eliminar el usuario.'], 500);
        }
    }
}

send_response(['success' => false, 'message' => 'Acción no válida.'], 400);