<?php
session_start();

// Si el usuario no ha iniciado sesión, redirigirlo a la página de login
if (!isset($_SESSION['user_logged_in']) || $_SESSION['user_logged_in'] !== true) {
    header('Location: login.html');
    exit;
}
?>
<!DOCTYPE html>
<html lang="es">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>Admin - Copesa Construcciones</title>

    <!-- No indexar esta página -->
    <meta name="robots" content="noindex, nofollow">

    <!-- Favicon -->
    <link rel="icon" type="image/png" href="/webcopesa/images/iso.png">
    
    <!-- Tailwind CSS CDN -->
    <script src="https://cdn.tailwindcss.com"></script>
    
    <!-- Google Font (Inter) -->
    <link rel="preconnect" href="https://fonts.googleapis.com">
    <link rel="preconnect" href="https://fonts.gstatic.com" crossorigin>
    <link href="https://fonts.googleapis.com/css2?family=Inter:wght@400;500;600;700&display=swap" rel="stylesheet">

    <style>
        body {
            font-family: 'Inter', sans-serif;
        }
        .admin-content {
            animation: fadeIn 0.3s ease-in-out;
        }
        @keyframes fadeIn {
            from { opacity: 0; transform: translateY(10px); }
            to { opacity: 1; transform: translateY(0); }
        }
        .sidebar-link.active {
            background-color: #f04e23;
            color: white;
        }
        .modal-backdrop {
            background-color: rgba(0, 0, 0, 0.5);
            transition: opacity 0.3s ease-in-out;
        }
    </style>

    <script>
        // Configuración de Tailwind para agregar los colores de la marca
        tailwind.config = {
            theme: {
                extend: {
                    colors: {
                        copesa: {
                            orange: '#f04e23',
                        }
                    }
                }
            }
        }
    </script>
</head>
<body class="bg-gray-100">

    <div class="flex h-screen bg-gray-200">
        <!-- Sidebar -->
        <div class="w-64 bg-gray-800 text-white flex flex-col">
            <div class="px-6 py-4 border-b border-gray-700">
                <h1 class="text-2xl font-bold text-copesa-orange">COPESA</h1>
                <span class="text-sm text-gray-400">Panel de Control</span>
            </div>
            <nav class="flex-1 px-4 py-4 space-y-2">
                <a href="#" class="sidebar-link admin-nav-trigger flex items-center px-4 py-2 rounded-md hover:bg-gray-700" data-page="mensajes">
                    <i data-lucide="inbox" class="w-5 h-5 mr-3"></i>
                    Mensajes Contacto
                </a>
                <a href="#" class="sidebar-link admin-nav-trigger flex items-center px-4 py-2 rounded-md hover:bg-gray-700" data-page="configuracion">
                    <i data-lucide="settings" class="w-5 h-5 mr-3"></i>
                    Configuración Keys
                </a>
                <a href="#" class="sidebar-link admin-nav-trigger flex items-center px-4 py-2 rounded-md hover:bg-gray-700" data-page="usuarios">
                    <i data-lucide="users" class="w-5 h-5 mr-3"></i>
                    Gestión de Usuarios
                </a>
                <a href="#" class="sidebar-link admin-nav-trigger flex items-center px-4 py-2 rounded-md hover:bg-gray-700" data-page="proyectos">
                    <i data-lucide="briefcase" class="w-5 h-5 mr-3"></i>
                    Gestión de Proyectos
                </a>
            </nav>
            <div class="px-6 py-4 border-t border-gray-700">
                <p class="text-sm text-gray-400 mb-2">Conectado como:</p>
                <p class="font-semibold text-white mb-4"><?php echo htmlspecialchars($_SESSION['user_name']); ?></p>
                
                <a href="php/logout_handler.php" class="flex items-center text-gray-400 hover:text-white text-sm">
                    <i data-lucide="log-out" class="w-4 h-4 mr-2"></i>
                    Cerrar Sesión
                </a>
            </div>
        </div>

        <!-- Main Content -->
        <main class="flex-1 p-6 md:p-10 overflow-y-auto">
            
            <!-- Página: Mensajes de Contacto -->
            <section id="page-mensajes" class="admin-content">
                <h2 class="text-3xl font-bold text-gray-800 mb-6">Mensajes del Formulario de Contacto</h2>
                <div class="bg-white rounded-lg shadow overflow-hidden">
                    <div id="messages-list" class="divide-y divide-gray-200">
                        <!-- Los mensajes se cargarán aquí dinámicamente -->
                        <p class="p-6 text-gray-500">Cargando mensajes...</p>
                    </div>
                </div>
            </section>

            <!-- Página: Configuración de API Keys -->
            <section id="page-configuracion" class="admin-content hidden">
                <h2 class="text-3xl font-bold text-gray-800 mb-6">Configuración de API Keys</h2>
                <div class="bg-white p-6 rounded-lg shadow space-y-8">
                    <!-- reCAPTCHA v3 -->
                    <div>
                        <h3 class="text-xl font-semibold mb-4 border-b pb-2">Google reCAPTCHA v3</h3>
                        <div class="space-y-4">
                            <div>
                                <label class="block text-sm font-medium text-gray-700">Site Key (Clave del sitio)</label>
                                <input type="text" id="recaptcha_site_key" name="recaptcha_site_key" placeholder="Pega aquí la Site Key de reCAPTCHA" class="mt-1 block w-full border-gray-300 rounded-md shadow-sm focus:ring-copesa-orange focus:border-copesa-orange">
                            </div>
                            <div>
                                <label class="block text-sm font-medium text-gray-700">Secret Key (Clave secreta)</label>
                                <input type="password" id="recaptcha_secret_key" name="recaptcha_secret_key" placeholder="Pega aquí la Secret Key de reCAPTCHA" class="mt-1 block w-full border-gray-300 rounded-md shadow-sm focus:ring-copesa-orange focus:border-copesa-orange">
                            </div>
                        </div>
                    </div>
                    <!-- Google Analytics -->
                    <div>
                        <h3 class="text-xl font-semibold mb-4 border-b pb-2">Google Analytics</h3>
                        <div class="space-y-4">
                            <div>
                                <label class="block text-sm font-medium text-gray-700">Measurement ID (ID de medición)</label>
                                <input type="text" id="ga_measurement_id" name="ga_measurement_id" placeholder="Pega aquí el ID de medición (ej. G-XXXXXXXXXX)" class="mt-1 block w-full border-gray-300 rounded-md shadow-sm focus:ring-copesa-orange focus:border-copesa-orange">
                            </div>
                        </div>
                    </div>
                    <div class="text-right">
                        <button id="save-config-btn" class="bg-copesa-orange text-white font-semibold px-6 py-2 rounded-lg shadow hover:bg-opacity-90">
                            Guardar Cambios
                        </button>
                    </div>
                </div>
            </section>

            <!-- Página: Gestión de Usuarios -->
            <section id="page-usuarios" class="admin-content hidden">
                <h2 class="text-3xl font-bold text-gray-800 mb-6">Gestión de Usuarios</h2>
                <div class="bg-white p-6 rounded-lg shadow">
                    <div class="grid grid-cols-1 lg:grid-cols-3 gap-8">
                        <!-- Formulario para crear usuario -->
                        <div class="lg:col-span-1">
                            <h3 id="user-form-title" class="text-xl font-semibold mb-4">Crear Nuevo Usuario</h3>
                            <form id="user-form" class="space-y-4">
                                <input type="hidden" id="user_id" name="user_id">
                                <div>
                                    <label for="full_name" class="block text-sm font-medium text-gray-700">Nombre Completo</label>
                                    <input type="text" id="full_name" name="full_name" required class="mt-1 block w-full border-gray-300 rounded-md shadow-sm focus:ring-copesa-orange focus:border-copesa-orange">
                                </div>
                                <div>
                                    <label for="username" class="block text-sm font-medium text-gray-700">Nombre de Usuario</label>
                                    <input type="text" id="username" name="username" required autocomplete="off" class="mt-1 block w-full border-gray-300 rounded-md shadow-sm focus:ring-copesa-orange focus:border-copesa-orange">
                                </div>
                                <div>
                                    <label for="password" class="block text-sm font-medium text-gray-700">Contraseña</label>
                                    <input type="password" id="password" name="password" required autocomplete="new-password" class="mt-1 block w-full border-gray-300 rounded-md shadow-sm focus:ring-copesa-orange focus:border-copesa-orange">
                                </div>
                                <button type="submit" id="user-form-submit-btn" class="w-full bg-copesa-orange text-white font-semibold px-5 py-2.5 rounded-lg shadow hover:bg-opacity-90 transition-all">Crear Usuario</button>
                                <button type="button" id="cancel-edit-btn" class="w-full bg-gray-200 text-gray-700 font-semibold px-5 py-2.5 rounded-lg hover:bg-gray-300 transition-all hidden">Cancelar Edición</button>
                            </form>
                        </div>
                        <!-- Lista de usuarios existentes -->
                        <div id="users-list-container" class="lg:col-span-2">
                            <!-- La lista de usuarios se cargará aquí -->
                        </div>
                    </div>
                </div>
            </section>

            <!-- Página: Gestión de Proyectos -->
            <section id="page-proyectos" class="admin-content hidden">
                <h2 class="text-3xl font-bold text-gray-800 mb-6">Gestión de Proyectos del Portafolio</h2>
                <div class="bg-white p-6 rounded-lg shadow">
                    <div class="grid grid-cols-1 lg:grid-cols-3 gap-8">
                        <!-- Formulario para crear/editar proyecto -->
                        <div class="lg:col-span-2">
                            <h3 id="project-form-title" class="text-xl font-semibold mb-4">Añadir Nuevo Proyecto</h3>
                            <form id="project-form" class="space-y-4" enctype="multipart/form-data">
                                <input type="hidden" id="project_id" name="id">
                                <input type="hidden" name="action" value="create">
                                <div>
                                    <label for="project_title" class="block text-sm font-medium text-gray-700">Título del Proyecto</label>
                                    <input type="text" id="project_title" name="project_title" required class="mt-1 block w-full border-gray-300 rounded-md shadow-sm focus:ring-copesa-orange focus:border-copesa-orange">
                                </div>
                                <div>
                                    <label for="project_subtitle" class="block text-sm font-medium text-gray-700">Subtítulo (Opcional)</label>
                                    <input type="text" id="project_subtitle" name="project_subtitle" class="mt-1 block w-full border-gray-300 rounded-md shadow-sm focus:ring-copesa-orange focus:border-copesa-orange">
                                </div>
                                <div>
                                    <label for="project_category" class="block text-sm font-medium text-gray-700">Categoría</label>
                                    <select id="project_category" name="project_category" required class="mt-1 block w-full border-gray-300 rounded-md shadow-sm focus:ring-copesa-orange focus:border-copesa-orange">
                                        <option value="">Selecciona una categoría</option>
                                        <option value="comercial">Comercial</option>
                                        <option value="residencial">Residencial</option>
                                        <option value="industrial">Industrial</option>
                                        <option value="corporativo">Corporativo</option>
                                        <option value="retail">Retail</option>
                                        <option value="salud">Salud</option>
                                    </select>
                                </div>
                                <div>
                                    <label for="project_description" class="block text-sm font-medium text-gray-700">Descripción Breve</label>
                                    <textarea id="project_description" name="project_description" rows="3" class="mt-1 block w-full border-gray-300 rounded-md shadow-sm focus:ring-copesa-orange focus:border-copesa-orange"></textarea>
                                </div>
                                <!-- Contenedor para imágenes actuales (en modo edición) -->
                                <div id="current-images-container" class="hidden">
                                    <label class="block text-sm font-medium text-gray-700">Imágenes Actuales</label>
                                    <div id="current-images-list" class="mt-2 grid grid-cols-3 gap-4"></div>
                                </div>
                                <div>
                                    <label for="project_images" class="block text-sm font-medium text-gray-700">Imágenes del Proyecto</label>
                                    <input type="file" id="project_images" name="project_images[]" accept="image/*" multiple class="mt-1 block w-full text-sm text-gray-500 file:mr-4 file:py-2 file:px-4 file:rounded-full file:border-0 file:text-sm file:font-semibold file:bg-orange-50 file:text-copesa-orange hover:file:bg-orange-100">
                                    <p id="project-images-help" class="text-xs text-gray-500 mt-1">Puedes seleccionar varias imágenes. La primera será la imagen de portada.</p>
                                </div>
                                <button type="submit" id="project-form-submit-btn" class="w-full bg-copesa-orange text-white font-semibold px-5 py-2.5 rounded-lg shadow hover:bg-opacity-90 transition-all">Guardar Proyecto</button>
                                <button type="button" id="cancel-project-edit-btn" class="w-full bg-gray-200 text-gray-700 font-semibold px-5 py-2.5 rounded-lg hover:bg-gray-300 transition-all hidden">Cancelar Edición</button>
                            </form>
                        </div>
                        <!-- Lista de proyectos existentes -->
                        <div class="lg:col-span-1">
                            <h3 class="text-xl font-semibold mb-4">Proyectos Existentes</h3>
                            <div id="projects-list" class="space-y-3">
                                <!-- La lista de proyectos se cargará aquí -->
                            </div>
                        </div>
                    </div>
                </div>
            </section>

        </main>
    </div>

    <!-- Modal para Ver Mensaje -->
    <div id="message-modal" class="fixed inset-0 z-50 hidden items-center justify-center p-4 modal-backdrop">
        <div class="bg-white rounded-lg shadow-xl w-full max-w-2xl max-h-full overflow-y-auto">
            <div class="p-6 border-b flex justify-between items-center">
                <h3 class="text-2xl font-bold">Detalles del Mensaje</h3>
                <button id="modal-close-btn" class="text-gray-500 hover:text-gray-800">
                    <i data-lucide="x" class="w-7 h-7"></i>
                </button>
            </div>
            <div class="p-6 space-y-4">
                <p><strong>De:</strong> <span id="modal-name"></span> &lt;<span id="modal-email"></span>&gt;</p>
                <p><strong>Fecha:</strong> <span id="modal-date"></span></p>
                <div>
                    <p class="font-bold mb-2">Mensaje:</p>
                    <div id="modal-message" class="bg-gray-100 p-4 rounded-md whitespace-pre-wrap"></div>
                </div>
            </div>
        </div>
    </div>

    <!-- Modal para Ver Usuario -->
    <div id="user-view-modal" class="fixed inset-0 z-50 hidden items-center justify-center p-4 modal-backdrop">
        <div class="bg-white rounded-lg shadow-xl w-full max-w-md">
            <div class="p-6 border-b flex justify-between items-center">
                <h3 class="text-2xl font-bold">Detalles del Usuario</h3>
                <button class="modal-close-btn text-gray-500 hover:text-gray-800">
                    <i data-lucide="x" class="w-7 h-7"></i>
                </button>
            </div>
            <div class="p-6 space-y-4">
                <p><strong>Nombre Completo:</strong> <span id="view-full_name"></span></p>
                <p><strong>Nombre de Usuario:</strong> <span id="view-username"></span></p>
            </div>
        </div>
    </div>

    <!-- Modal para Ver Proyecto -->
    <div id="project-view-modal" class="fixed inset-0 z-50 hidden items-center justify-center p-4 modal-backdrop">
        <div class="bg-white rounded-lg shadow-xl w-full max-w-3xl max-h-full overflow-y-auto">
            <div class="p-6 border-b flex justify-between items-center">
                <h3 class="text-2xl font-bold">Detalles del Proyecto</h3>
                <button class="modal-close-btn text-gray-500 hover:text-gray-800">
                    <i data-lucide="x" class="w-7 h-7"></i>
                </button>
            </div>
            <div id="project-modal-body" class="p-6 space-y-4">
                <p><strong>Título:</strong> <span id="view-project-title"></span></p>
                <p><strong>Subtítulo:</strong> <span id="view-project-subtitle"></span></p>
                <p><strong>Categoría:</strong> <span id="view-project-category"></span></p>
                <div>
                    <p class="font-bold mb-2">Descripción:</p>
                    <div id="view-project-description" class="bg-gray-100 p-4 rounded-md whitespace-pre-wrap"></div>
                </div>
                <div>
                    <p class="font-bold mb-2">Imágenes:</p>
                    <div id="view-project-images" class="grid grid-cols-2 sm:grid-cols-3 gap-4"></div>
                </div>
            </div>
        </div>
    </div>

    <script src="https://unpkg.com/lucide@latest"></script>
    <script>
        document.addEventListener("DOMContentLoaded", function() {
            const modal = document.getElementById('message-modal');
            const modalCloseBtn = document.getElementById('modal-close-btn');

            const showAdminPage = async (pageId) => {
                // Ocultar todas las secciones
                document.querySelectorAll('.admin-content').forEach(section => {
                    section.classList.add('hidden');
                });

                // Mostrar la sección solicitada
                const activePage = document.getElementById('page-' + pageId);
                if (activePage) {
                    activePage.classList.remove('hidden');
                }

                // Actualizar clase 'active' en el sidebar
                document.querySelectorAll('.sidebar-link').forEach(link => {
                    link.classList.remove('active');
                });
                const activeLink = document.querySelector(`.admin-nav-trigger[data-page="${pageId}"]`);
                if (activeLink) {
                    activeLink.classList.add('active');
                }

                // Si la página es la de mensajes, cargar los datos
                if (pageId === 'mensajes') {
                    await loadMessages();
                }

                // Si la página es la de configuración, cargar las keys
                if (pageId === 'configuracion') {
                    await loadConfig();
                }

                // Si la página es la de usuarios, cargar los datos
                if (pageId === 'usuarios') {
                    await loadUsers();
                }

                // Si la página es la de proyectos, cargar los datos
                if (pageId === 'proyectos') {
                    await loadProjects();
                }
            };

            // --- Cargar y mostrar mensajes ---
            const loadMessages = async () => {
                const listContainer = document.getElementById('messages-list');
                listContainer.innerHTML = '<p class="p-6 text-gray-500">Cargando mensajes...</p>';

                try {
                    // Usamos un parámetro 'cache-buster' para evitar que el navegador use una versión antigua del archivo
                    const response = await fetch(`/webcopesa/data/messages.json?v=${new Date().getTime()}`);
                    
                    if (!response.ok) {
                        // Si el archivo no existe (404), es normal la primera vez.
                        if (response.status === 404) {
                            listContainer.innerHTML = '<p class="p-6 text-gray-500">Aún no se han recibido mensajes.</p>';
                            return;
                        }
                        throw new Error('No se pudo cargar el archivo de mensajes.');
                    }

                    const messages = await response.json();

                    if (!messages || messages.length === 0) {
                        listContainer.innerHTML = '<p class="p-6 text-gray-500">Aún no se han recibido mensajes.</p>';
                        return;
                    }

                    // Limpiar contenedor y renderizar mensajes
                    listContainer.innerHTML = '';
                    messages.forEach(msg => {
                        const messageElement = document.createElement('div');
                        // Añadir clase si el mensaje no está leído
                        const isUnread = msg.read === false ? 'bg-orange-100' : 'bg-white';
                        messageElement.className = `p-4 sm:p-6 hover:bg-gray-50 transition-colors ${isUnread}`;
                        
                        messageElement.innerHTML = `
                            <div class="flex flex-col sm:flex-row justify-between sm:items-center">
                                <div>
                                    <h4 class="font-bold text-lg text-gray-800">${msg.name}</h4>
                                    <span class="text-sm text-gray-500">${msg.email}</span>
                                    <span class="block sm:inline text-xs text-gray-400 sm:ml-2">${msg.date}</span>
                                </div>
                                <div class="flex items-center space-x-2 mt-4 sm:mt-0">
                                    <button class="view-btn text-sm bg-blue-500 hover:bg-blue-600 text-white font-semibold py-1 px-3 rounded-md transition-colors" data-id="${msg.id}">Ver</button>
                                    <button class="delete-btn text-sm bg-red-500 hover:bg-red-600 text-white font-semibold py-1 px-3 rounded-md transition-colors" data-id="${msg.id}">Eliminar</button>
                                </div>
                            </div>
                            <p class="text-gray-600 mt-3 truncate">${msg.message}</p>`;
                        
                        // Guardar los datos completos en el elemento para usarlos en el modal
                        messageElement.dataset.fullMessage = JSON.stringify(msg);
                        listContainer.appendChild(messageElement);
                    });
                } catch (error) {
                    console.error('Error cargando mensajes:', error);
                    listContainer.innerHTML = `<p class="p-6 text-red-600">Error al cargar los mensajes: ${error.message}</p>`;
                }
            };

            // --- Lógica del Modal ---
            const openMessageModal = (msg) => {
                document.getElementById('modal-name').textContent = msg.name;
                document.getElementById('modal-email').textContent = msg.email;
                document.getElementById('modal-date').textContent = msg.date;
                document.getElementById('modal-message').textContent = msg.message;
                modal.classList.remove('hidden');
                modal.classList.add('flex');
            };

            const closeMessageModal = () => {
                modal.classList.add('hidden');
                modal.classList.remove('flex');
            };

            modalCloseBtn.addEventListener('click', closeMessageModal);
            modal.addEventListener('click', (e) => {
                if (e.target === modal) { // Cerrar si se hace clic en el fondo
                    closeMessageModal();
                }
            });

            // --- Lógica para Ver y Eliminar ---
            document.getElementById('messages-list').addEventListener('click', async (e) => {
                const target = e.target;

                // Botón VER
                if (target.classList.contains('view-btn')) {
                    const messageData = JSON.parse(target.closest('[data-full-message]').dataset.fullMessage);
                    openMessageModal(messageData);

                    // Marcar como leído si no lo está
                    if (messageData.read === false) {
                        await fetch('/webcopesa/php/marcar_leido.php', {
                            method: 'POST',
                            headers: { 'Content-Type': 'application/json' },
                            body: JSON.stringify({ id: messageData.id })
                        });
                        // Recargar la lista para que el color de fondo se actualice
                        await loadMessages();
                    }
                }

                // Botón ELIMINAR
                if (target.classList.contains('delete-btn')) {
                    const messageId = target.dataset.id;
                    if (confirm('¿Estás seguro de que quieres eliminar este mensaje? Esta acción no se puede deshacer.')) {
                        try {
                            const response = await fetch('/webcopesa/eliminar_mensaje.php', {
                                method: 'POST',
                                headers: { 'Content-Type': 'application/json' },
                                body: JSON.stringify({ id: messageId })
                            });
                            const result = await response.json();
                            if (result.success) {
                                await loadMessages(); // Recargar la lista de mensajes
                            } else {
                                alert('Error: ' + result.message);
                            }
                        } catch (error) {
                            alert('Error de conexión al intentar eliminar el mensaje.');
                        }
                    }
                }
            });

            // --- Lógica para Configuración de API Keys ---
            const loadConfig = async () => {
                try {
                    const response = await fetch('/webcopesa/php/config_handler.php');
                    const config = await response.json();
                    document.getElementById('recaptcha_site_key').value = config.recaptcha_site_key || '';
                    document.getElementById('ga_measurement_id').value = config.ga_measurement_id || '';
                    // La clave secreta no se carga por seguridad, solo se puede escribir.
                    document.getElementById('recaptcha_secret_key').value = '';
                    document.getElementById('recaptcha_secret_key').placeholder = 'Dejar en blanco para no cambiar';
                } catch (error) {
                    console.error('Error al cargar la configuración:', error);
                    alert('No se pudo cargar la configuración de las API Keys.');
                }
            };

            const saveConfigBtn = document.getElementById('save-config-btn');
            if (saveConfigBtn) {
                saveConfigBtn.addEventListener('click', async () => {
                    const siteKey = document.getElementById('recaptcha_site_key').value;
                    const secretKey = document.getElementById('recaptcha_secret_key').value;
                    const gaId = document.getElementById('ga_measurement_id').value;

                    const configData = {
                        recaptcha_site_key: siteKey,
                        ga_measurement_id: gaId,
                    };

                    // Solo incluir la secret key si el usuario ha escrito algo
                    if (secretKey) {
                        configData.recaptcha_secret_key = secretKey;
                    }

                    const response = await fetch('/webcopesa/php/config_handler.php', {
                        method: 'POST',
                        headers: { 'Content-Type': 'application/json' },
                        body: JSON.stringify(configData)
                    });
                    const result = await response.json();
                    alert(result.message);
                    if (result.success) {
                        await loadConfig(); // Recargar para limpiar el campo de la secret key
                    }
                });
            }

            // --- Lógica para Gestión de Usuarios ---
            const usersListContainer = document.getElementById('users-list-container');
            const userForm = document.getElementById('user-form');
            let allUsersData = []; // Caché para los datos de los usuarios
            const loadUsers = async () => {
                if (!usersListContainer) return;
                usersListContainer.innerHTML = '<p class="text-gray-500">Cargando usuarios...</p>';
                try {
                    const response = await fetch('/webcopesa/data/user_handler.php');
                    const users = await response.json();

                    allUsersData = users; // Guardar datos en caché
                    let userListHtml = '<h3 class="text-xl font-semibold mb-4">Usuarios Existentes</h3>';
                    if (users.length === 0) {
                        userListHtml += '<p class="text-gray-500">No hay usuarios creados.</p>';
                    } else {
                        userListHtml += '<div class="space-y-3">';
                        users.forEach(user => {
                            userListHtml += `
                                <div class="flex justify-between items-center bg-gray-50 p-3 rounded-md">
                                    <div>
                                        <p class="font-semibold text-gray-800">${user.full_name}</p>
                                        <p class="text-sm text-gray-500">@${user.username}</p>
                                    </div>
                                    <div class="flex items-center space-x-3">
                                        <button class="view-user-btn text-blue-500 hover:text-blue-700" data-id="${user.id}" title="Ver usuario"><i data-lucide="eye" class="w-5 h-5 pointer-events-none"></i></button>
                                        <button class="edit-user-btn text-green-600 hover:text-green-800" data-id="${user.id}" title="Editar usuario"><i data-lucide="file-pen-line" class="w-5 h-5 pointer-events-none"></i></button>
                                        <button class="delete-user-btn text-red-500 hover:text-red-700" data-id="${user.id}" title="Eliminar usuario"><i data-lucide="trash-2" class="w-5 h-5 pointer-events-none"></i></button>
                                    </div>
                                </div>`;
                        });
                        userListHtml += '</div>';
                    }
                    usersListContainer.innerHTML = userListHtml;
                    lucide.createIcons(); // Re-renderizar íconos de Lucide
                } catch (error) {
                    usersListContainer.innerHTML = '<p class="text-red-600">Error al cargar los usuarios.</p>';
                }
            };

            const resetUserForm = () => {
                userForm.reset();
                document.getElementById('user_id').value = '';
                document.getElementById('user-form-title').textContent = 'Crear Nuevo Usuario';
                document.getElementById('user-form-submit-btn').textContent = 'Crear Usuario';
                document.getElementById('password').setAttribute('required', 'true');
                document.getElementById('password').placeholder = '';
                document.getElementById('cancel-edit-btn').classList.add('hidden');
            };

            if (userForm) {
                userForm.addEventListener('submit', async (e) => {
                    e.preventDefault();
                    const formData = new FormData(userForm);
                    const data = Object.fromEntries(formData.entries());
                    const userId = document.getElementById('user_id').value;

                    data.action = userId ? 'edit' : 'create';
                    if (userId) {
                        data.id = userId;
                    }
                    // No enviar contraseña vacía en modo edición
                    if (data.action === 'edit' && !data.password) {
                        delete data.password;
                    }

                    try {
                        const response = await fetch('/webcopesa/data/user_handler.php', {
                            method: 'POST',
                            headers: { 'Content-Type': 'application/json' },
                            body: JSON.stringify(data)
                        });
                        const result = await response.json();
                        alert(result.message);
                        if (result.success) {
                            resetUserForm();
                            await loadUsers();
                        }
                    } catch (error) {
                        alert('Error de conexión al guardar el usuario.');
                    }
                });

                document.getElementById('cancel-edit-btn').addEventListener('click', resetUserForm);
            }

            if (usersListContainer) {
                usersListContainer.addEventListener('click', async (e) => {
                    const target = e.target;
                    const userId = target.closest('button')?.dataset.id;
                    if (!userId) return;

                    // Botón VER
                    if (target.closest('.view-user-btn')) {
                        const user = allUsersData.find(u => u.id === userId);
                        if (user) {
                            document.getElementById('view-full_name').textContent = user.full_name;
                            document.getElementById('view-username').textContent = user.username;
                            document.getElementById('user-view-modal').classList.remove('hidden');
                            document.getElementById('user-view-modal').classList.add('flex');
                        }
                    }

                    // Botón EDITAR
                    if (target.closest('.edit-user-btn')) {
                        const user = allUsersData.find(u => u.id === userId);
                        if (user) {
                            document.getElementById('user_id').value = user.id;
                            document.getElementById('full_name').value = user.full_name;
                            document.getElementById('username').value = user.username;
                            document.getElementById('password').value = '';
                            document.getElementById('password').removeAttribute('required');
                            document.getElementById('password').placeholder = 'Dejar en blanco para no cambiar';
                            document.getElementById('user-form-title').textContent = 'Editar Usuario';
                            document.getElementById('user-form-submit-btn').textContent = 'Guardar Cambios';
                            document.getElementById('cancel-edit-btn').classList.remove('hidden');
                            userForm.scrollIntoView({ behavior: 'smooth' });
                        }
                    }

                    // Botón ELIMINAR
                    if (target.closest('.delete-user-btn')) {
                        if (confirm('¿Estás seguro de que quieres eliminar este usuario?')) {
                            const response = await fetch('/webcopesa/data/user_handler.php', {
                                method: 'POST',
                                headers: { 'Content-Type': 'application/json' },
                                body: JSON.stringify({ action: 'delete', id: userId })
                            });
                            const result = await response.json();
                            alert(result.message);
                            if (result.success) {
                                resetUserForm(); // Limpiar formulario si se estaba editando el usuario eliminado
                                await loadUsers();
                            }
                        }
                    }
                });
            }

            // --- Lógica Genérica para Cerrar Modales ---
            // Cerrar con el botón X
            document.querySelectorAll('.modal-close-btn').forEach(btn => {
                btn.addEventListener('click', () => {
                    btn.closest('.modal-backdrop').classList.add('hidden');
                    btn.closest('.modal-backdrop').classList.remove('flex');
                });
            });
            // Cerrar haciendo clic en el fondo oscuro
            document.addEventListener('click', (e) => {
                if (e.target.classList.contains('modal-backdrop')) {
                    e.target.classList.add('hidden');
                    e.target.classList.remove('flex');
                }
            });

            // --- Lógica para Gestión de Proyectos ---
            const projectForm = document.getElementById('project-form');
            const projectsListContainer = document.getElementById('projects-list');
            let allProjectsData = [];

            const loadProjects = async () => {
                if (!projectsListContainer) return;
                projectsListContainer.innerHTML = '<p class="text-gray-500">Cargando proyectos...</p>';
                try {
                    const response = await fetch(`/webcopesa/data/project_handler.php?v=${new Date().getTime()}`);
                    const projects = await response.json();
                    
                    allProjectsData = projects;
                    if (projects.length === 0) {
                        projectsListContainer.innerHTML = '<p class="text-gray-500">No hay proyectos creados.</p>';
                        return;
                    }

                    let listHtml = '';
                    projects.forEach(proj => {
                        listHtml += `
                            <div class="flex items-center justify-between bg-gray-50 p-3 rounded-md">
                                <p class="font-semibold text-sm truncate pr-2" title="${proj.title}">${proj.title}</p>
                                <div class="flex items-center space-x-2 flex-shrink-0">
                                    <button class="view-project-btn text-blue-500 hover:text-blue-700" data-id="${proj.id}" title="Ver proyecto"><i data-lucide="eye" class="w-4 h-4 pointer-events-none"></i></button>
                                    <button class="edit-project-btn text-green-600 hover:text-green-800" data-id="${proj.id}" title="Editar proyecto"><i data-lucide="file-pen-line" class="w-4 h-4 pointer-events-none"></i></button>
                                    <button class="delete-project-btn text-red-500 hover:text-red-700" data-id="${proj.id}" title="Eliminar proyecto">
                                        <i data-lucide="trash-2" class="w-4 h-4 pointer-events-none"></i>
                                    </button>
                                </div>
                            </div>`;
                    });
                    projectsListContainer.innerHTML = listHtml;
                    lucide.createIcons();
                } catch (error) {
                    projectsListContainer.innerHTML = '<p class="text-red-600">Error al cargar los proyectos.</p>';
                }
            };

            const resetProjectForm = () => {
                projectForm.reset();
                document.getElementById('project_id').value = '';
                projectForm.querySelector('[name="action"]').value = 'create';
                document.getElementById('project-form-title').textContent = 'Añadir Nuevo Proyecto';
                document.getElementById('project-form-submit-btn').textContent = 'Guardar Proyecto';
                document.getElementById('cancel-project-edit-btn').classList.add('hidden');
                document.getElementById('project_images').setAttribute('required', 'true');
                document.getElementById('current-images-container').classList.add('hidden');
                document.getElementById('current-images-list').innerHTML = '';
                document.getElementById('project-images-help').textContent = 'Puedes seleccionar varias imágenes. La primera será la imagen de portada.';
            };

            if (projectForm) {
                projectForm.addEventListener('submit', async (e) => {
                    e.preventDefault();
                    const formData = new FormData(projectForm);
                    const action = document.getElementById('project_id').value ? 'edit' : 'create';
                    formData.set('action', action);
                    
                    try {
                        // Si es edición y no se suben nuevas imágenes, eliminamos el campo para no dar error
                        if (action === 'edit' && document.getElementById('project_images').files.length === 0) {
                            formData.delete('project_images[]');
                        }

                        const response = await fetch('/webcopesa/data/project_handler.php', {
                            method: 'POST',
                            body: formData // No se usa JSON.stringify para FormData
                        });
                        const result = await response.json();
                        alert(result.message);
                        if (result.success) {
                            resetProjectForm();
                            await loadProjects();
                        }
                    } catch (error) {
                        alert('Error de conexión al guardar el proyecto.');
                    }
                });

                document.getElementById('cancel-project-edit-btn').addEventListener('click', resetProjectForm);
            }

            if (projectsListContainer) {
                projectsListContainer.addEventListener('click', async (e) => {
                    const btn = e.target.closest('button');
                    if (!btn) return;

                    const projectId = btn.dataset.id;

                    // Botón VER
                    if (btn.classList.contains('view-project-btn')) {
                        const project = allProjectsData.find(p => p.id === projectId);
                        if (project) {
                            document.getElementById('view-project-title').textContent = project.title;
                            document.getElementById('view-project-subtitle').textContent = project.subtitle || 'N/A';
                            document.getElementById('view-project-category').textContent = project.category;
                            document.getElementById('view-project-description').textContent = project.description || 'N/A';
                            const imagesContainer = document.getElementById('view-project-images');
                            imagesContainer.innerHTML = '';
                            project.images.forEach(imgSrc => {
                                imagesContainer.innerHTML += `<img src="${imgSrc}" class="w-full h-auto rounded-md shadow">`;
                            });
                            document.getElementById('project-view-modal').classList.remove('hidden');
                            document.getElementById('project-view-modal').classList.add('flex');
                        }
                    }

                    // Botón EDITAR
                    if (btn.classList.contains('edit-project-btn')) {
                        const project = allProjectsData.find(p => p.id === projectId);
                        if (project) {
                            document.getElementById('project_id').value = project.id;
                            document.getElementById('project_title').value = project.title;
                            document.getElementById('project_subtitle').value = project.subtitle;
                            document.getElementById('project_category').value = project.category;
                            document.getElementById('project_description').value = project.description;
                            document.getElementById('project_images').removeAttribute('required');
                            document.getElementById('project-images-help').textContent = 'Sube imágenes adicionales si lo deseas.';
                            document.getElementById('project-form-title').textContent = 'Editar Proyecto';
                            document.getElementById('project-form-submit-btn').textContent = 'Actualizar Proyecto';
                            document.getElementById('cancel-project-edit-btn').classList.remove('hidden');

                            // Mostrar imágenes actuales con opción de eliminar
                            const imagesContainer = document.getElementById('current-images-list');
                            imagesContainer.innerHTML = '';
                            project.images.forEach(imgSrc => {
                                imagesContainer.innerHTML += `
                                    <div class="relative group">
                                        <img src="${imgSrc}" class="w-full h-24 object-cover rounded-md">
                                        <input type="hidden" name="kept_images[]" value="${imgSrc}">
                                        <button type="button" class="absolute top-1 right-1 bg-red-600 text-white rounded-full p-0.5 opacity-75 group-hover:opacity-100" onclick="this.parentElement.remove()">
                                            <i data-lucide="x" class="w-3 h-3"></i>
                                        </button>
                                    </div>`;
                            });
                            document.getElementById('current-images-container').classList.remove('hidden');
                            lucide.createIcons();
                            projectForm.scrollIntoView({ behavior: 'smooth' });
                        }
                    }

                    // Botón ELIMINAR
                    if (btn.classList.contains('delete-project-btn')) {
                        const projectId = btn.dataset.id;

                        if (confirm('¿Estás seguro de que quieres eliminar este proyecto? Se borrarán también todas sus imágenes.')) {
                            const formData = new FormData();
                            formData.append('action', 'delete');
                            formData.append('id', projectId);

                            try {
                                const response = await fetch('/webcopesa/data/project_handler.php', {
                                    method: 'POST',
                                    body: formData
                                });
                                const result = await response.json();
                                alert(result.message);
                                if (result.success) {
                                    await loadProjects();
                                }
                            } catch (error) {
                                alert('Error de conexión al intentar eliminar el proyecto.');
                            }
                        }
                    }
                });
            }

            // Lógica de navegación del panel
            document.querySelectorAll('.admin-nav-trigger').forEach(trigger => {
                trigger.addEventListener('click', function(e) {
                    e.preventDefault();
                    const pageId = this.getAttribute('data-page');
                    if (pageId) {
                        showAdminPage(pageId);
                    }
                });
            });

            // Mostrar la primera página por defecto
            showAdminPage('mensajes'); // Ahora es una función asíncrona

            // Inicializar íconos
            lucide.createIcons();
        });
    </script>
</body>
</html>