$(document).ready(function() {
    const apiUrl = 'api/usuarios.php';
    const usuarioModal = new bootstrap.Modal(document.getElementById('usuarioModal'));
    const usuarioForm = $('#usuarioForm');
    const tablaUsuarios = $('#tablaUsuarios');
    const modalTitle = $('#usuarioModalLabel');
    const roleFieldContainer = $('#roleFieldContainer'); // The div containing the role select
    const roleSelect = $('#role'); // The role select itself
    const currentUserRole = $('#currentUserRole').val(); // Get current logged-in user's role

    let usersCache = []; // Cache para guardar la lista de usuarios

    // Función para cargar y mostrar usuarios
    function cargarUsuarios() {
        $.get(apiUrl, function(users) {
            usersCache = users; // Guardar en cache
            tablaUsuarios.empty();
            if (users.length === 0) {
                tablaUsuarios.append('<tr><td colspan="6" class="text-center">No hay usuarios registrados.</td></tr>');
            } else {
                users.forEach(user => {
                    tablaUsuarios.append(`
                        <tr data-id="${user.id}">
                            <td><img src="${user.foto || 'https://media.istockphoto.com/id/1332100919/es/vector/icono-de-hombre-icono-negro-s%C3%ADmbolo-de-persona.jpg?s=612x612&w=0&k=20&c=zf8iV9whu1NbDpw0H9A043wWjqyhPFP5m6CuesrBVFk='}" alt="Foto de ${user.nombreCompleto}" class="rounded-circle" width="50" height="50" style="object-fit: cover;"></td>
                            <td>${user.nombreCompleto}</td>
                            <td>${user.email}</td>
                            <td>${user.telefono || 'N/A'}</td>
                            <td>${user.role || 'N/A'}</td>
                            <td>
                                <button class="btn btn-sm btn-info btn-editar">Editar</button>
                                <button class="btn btn-sm btn-danger btn-eliminar">Eliminar</button>
                            </td>
                        </tr>
                    `);
                });
            }
        }).fail(function() {
            tablaUsuarios.empty().append('<tr><td colspan="6" class="text-center text-danger">Error al cargar los usuarios. Asegúrese de que el backend (PHP) está funcionando correctamente.</td></tr>');
        });
    }

    // Función para mostrar el modal de usuario (añadir/editar)
    function showUserModal(userToEdit = null) {
        usuarioForm[0].reset();
        usuarioForm.removeData('editing-id');

        // Control de visibilidad y editabilidad del campo de rol
        if (currentUserRole === 'admin') {
            roleFieldContainer.show(); // Mostrar el contenedor del campo de rol
            roleSelect.prop('disabled', false); // Habilitar el select
        } else {
            roleFieldContainer.hide(); // Ocultar el contenedor del campo de rol
            roleSelect.prop('disabled', true); // Deshabilitar el select (aunque esté oculto)
        }

        if (userToEdit) {
            modalTitle.text('Editar Usuario');
            $('#nombreCompleto').val(userToEdit.nombreCompleto);
            $('#email').val(userToEdit.email);
            $('#password').val(''); // No pre-rellenar contraseña por seguridad
            $('#direccion').val(userToEdit.direccion);
            $('#telefono').val(userToEdit.telefono);
            $('#fechaNacimiento').val(userToEdit.fechaNacimiento);
            $('#foto').val(userToEdit.foto);
            $('#hourly_rate').val(userToEdit.hourly_rate);
            roleSelect.val(userToEdit.role || 'team_member'); // Establecer el rol del usuario

            // Set the email notifications switch. Default to true if the property doesn't exist.
            $('#emailNotifications').prop('checked', userToEdit.email_notifications_enabled !== false);
            
            usuarioForm.data('editing-id', userToEdit.id);
        } else {
            modalTitle.text('Añadir Nuevo Usuario');
            // Establecer rol por defecto para nuevos usuarios si no es admin
            if (currentUserRole !== 'admin') {
                roleSelect.val('team_member');
            } else {
                roleSelect.val('team_member'); // Default for admin too
            }
            $('#emailNotifications').prop('checked', true); // Default to checked for new users
        }
        usuarioModal.show();
    }

    // Evento para abrir modal (AÑADIR)
    $('[data-bs-target="#usuarioModal"]').on('click', function() {
        showUserModal(null);
    });

    // Manejar el envío del formulario (Crear y Editar)
    usuarioForm.on('submit', function(e) {
        e.preventDefault();
        const editingId = usuarioForm.data('editing-id');

        const userData = {
            nombreCompleto: $('#nombreCompleto').val(),
            email: $('#email').val(),
            password: $('#password').val(),
            direccion: $('#direccion').val(),
            telefono: $('#telefono').val(),
            fechaNacimiento: $('#fechaNacimiento').val(),
            foto: $('#foto').val() || 'https://media.istockphoto.com/id/1332100919/es/vector/icono-de-hombre-icono-negro-s%C3%ADmbolo-de-persona.jpg?s=612x612&w=0&k=20&c=zf8iV9whu1NbDpw0H9A043wWjqyhPFP5m6CuesrBVFk=',
            hourly_rate: parseFloat($('#hourly_rate').val()) || 0,
            email_notifications_enabled: $('#emailNotifications').is(':checked')
        };

        // Solo incluir el rol si el campo es visible/editable (es decir, si el usuario actual es admin)
        if (currentUserRole === 'admin') {
            userData.role = roleSelect.val();
        } else if (editingId) {
            // Si no es admin y está editando, no enviar el rol para que el backend lo preserve
            // El backend ya maneja esto con el ?? $user['role'] ?? 'team_member'
        }

        const ajaxSettings = {
            url: editingId ? `${apiUrl}?id=${editingId}` : apiUrl,
            type: editingId ? 'PUT' : 'POST',
            contentType: 'application/json',
            data: JSON.stringify(userData),
            success: function() {
                usuarioModal.hide();
                cargarUsuarios();
            },
            error: function(xhr) {
                let errorMessage = 'Error al guardar el usuario.';
                if (xhr.responseJSON && xhr.responseJSON.error) {
                    errorMessage = xhr.responseJSON.error;
                }
                alert(errorMessage);
            }
        };

        $.ajax(ajaxSettings);
    });

    // Delegar evento para EDITAR
    tablaUsuarios.on('click', '.btn-editar', function() {
        const userId = $(this).closest('tr').data('id');
        const userToEdit = usersCache.find(u => u.id === userId);
        if (userToEdit) {
            showUserModal(userToEdit);
        }
    });

    // Delegar evento para ELIMINAR
    tablaUsuarios.on('click', '.btn-eliminar', function() {
        // Solo permitir eliminar si el usuario actual es admin
        if (currentUserRole !== 'admin') {
            alert('Permiso denegado. Solo los administradores pueden eliminar usuarios.');
            return;
        }

        const userId = $(this).closest('tr').data('id');
        
        if (confirm('¿Estás seguro de que quieres eliminar este usuario?')) {
            $.ajax({
                url: `${apiUrl}?id=${userId}`,
                type: 'DELETE',
                success: function() {
                    cargarUsuarios();
                },
                error: function(xhr) {
                    let errorMessage = 'Error al eliminar el usuario.';
                    if (xhr.responseJSON && xhr.responseJSON.error) {
                        errorMessage = xhr.responseJSON.error;
                    }
                    alert(errorMessage);
                }
            });
        }
    });

    // Cargar usuarios al iniciar la página
    cargarUsuarios();
});