$(document).ready(function() {
    const tasksApiUrl = 'api/tasks.php';
    const usersApiUrl = 'api/usuarios.php';
    const projectsApiUrl = 'api/projects.php';
    const commentsApiUrl = 'api/comments.php';

    const taskModal = new bootstrap.Modal(document.getElementById('taskModal'));
    const taskForm = $('#taskForm');
    const ganttElement = $('#gantt');
    const ganttTitle = $('#gantt-title');
    const taskListBody = $('#taskListBody');

    const urlParams = new URLSearchParams(window.location.search);
    const currentUserRole = $('#currentUserRole').val();
    const projectIdFromUrl = urlParams.get('project_id');

    let gantt;
    let allTasks = [];
    let allUsers = [];
    let allProjects = [];
    let currentTaskWorkLogs = [];
    let tasksById = {};

    function formatDateToDDMMYYYY(dateString) {
        if (!dateString) return 'N/A';
        const [year, month, day] = dateString.split('-');
        return `${day}/${month}/${year}`;
    }

    function formatDateTimeToDDMMYYYYHHMM(dateTimeString) {
        if (!dateTimeString) return 'N/A';
        const date = new Date(dateTimeString);
        const day = String(date.getDate()).padStart(2, '0');
        const month = String(date.getMonth() + 1).padStart(2, '0');
        const year = date.getFullYear();
        const hours = String(date.getHours()).padStart(2, '0');
        const minutes = String(date.getMinutes()).padStart(2, '0');
        return `${day}/${month}/${year} ${hours}:${minutes}`;
    }

    function formatNumber(value) {
        if (typeof value !== 'number') {
            value = parseFloat(value);
        }
        if (isNaN(value)) {
            return 'N/A';
        }
        return value.toLocaleString('es-PY', { minimumFractionDigits: 0, maximumFractionDigits: 2 });
    }

    function calculateHours(startTime, endTime) {
        if (!startTime || !endTime) return 0;
        const [startH, startM] = startTime.split(':').map(Number);
        const [endH, endM] = endTime.split(':').map(Number);
        const startDate = new Date(0, 0, 0, startH, startM);
        const endDate = new Date(0, 0, 0, endH, endM);
        let diff = endDate.getTime() - startDate.getTime();
        if (diff < 0) { diff += 1000 * 60 * 60 * 24; }
        return formatNumber(diff / (1000 * 60 * 60));
    }

    function renderWorkLogs(workLogs) {
        const workLogsList = $('#workLogsList');
        workLogsList.empty();
        if (!workLogs || workLogs.length === 0) {
            workLogsList.append('<p class="text-muted text-center mt-3">No hay trabajos registrados para esta tarea.</p>');
            return;
        }
        workLogs.forEach(log => {
            const logHtml = `
                <div class="list-group-item list-group-item-action flex-column align-items-start mb-2">
                    <div class="d-flex w-100 justify-content-between">
                        <h6 class="mb-1">${log.type} - ${formatDateToDDMMYYYY(log.date)} (${formatNumber(log.hours)} horas)</h6>
                        <small>
                            <button type="button" class="btn btn-sm btn-outline-primary edit-work-log" data-id="${log.id}">Editar</button>
                            <button type="button" class="btn btn-sm btn-outline-danger delete-work-log" data-id="${log.id}">Eliminar</button>
                        </small>
                    </div>
                    <p class="mb-1">${log.description}</p>
                    ${log.link ? `<small><a href="${log.link}" target="_blank">Ver archivos</a></small>` : ''}
                </div>
            `;
            workLogsList.append(logHtml);
        });
    }

    function calculateAndUpdateParentProgress() {
        tasksById = allTasks.reduce((acc, task) => {
            acc[task.id] = { ...task, children: [] };
            return acc;
        }, {});

        Object.values(tasksById).forEach(task => {
            if (task.parent_id && tasksById[task.parent_id]) {
                tasksById[task.parent_id].children.push(task);
            }
        });

        const parents = Object.values(tasksById).filter(t => t.children.length > 0);
        
        // Iterar varias veces para asegurar que los cambios se propaguen hacia arriba en la jerarquía
        for (let i = 0; i < 5; i++) { 
            parents.forEach(parent => {
                let totalProgress = 0;
                parent.children.forEach(child => {
                    totalProgress += parseInt(tasksById[child.id].progress, 10) || 0;
                });
                const avgProgress = Math.round(totalProgress / parent.children.length);
                tasksById[parent.id].progress = avgProgress;
            });
        }
        allTasks = Object.values(tasksById);
    }

    function init() {
        const getTasks = $.get(tasksApiUrl);
        const getUsers = $.get(usersApiUrl);
        const getProjects = $.get(projectsApiUrl);

        Promise.all([getTasks, getUsers, getProjects]).then(function([tasks, users, projects]) {
            allTasks = tasks;
            allUsers = users;
            allProjects = projects;

            calculateAndUpdateParentProgress();

            populateUserDropdown(users);
            populateProjectDropdown(projects);
            
            renderTaskList(allTasks);
            const tasksForGantt = JSON.parse(JSON.stringify(allTasks));
            initializeGantt(tasksForGantt);
            setupEventListeners();

            if (projectIdFromUrl) {
                const projectToFilter = allProjects.find(p => p.id === projectIdFromUrl);
                if (projectToFilter) {
                    $('#filterTaskProject').val(projectToFilter.nombre);
                    ganttTitle.text(`Proyecto: ${projectToFilter.nombre}`);
                    filterTasks();
                } else {
                    ganttTitle.text('Proyecto no encontrado');
                }
            } else {
                ganttTitle.text('Gantt de Todos los Proyectos');
            }

        }).catch(function(err) {
            console.error("Error al cargar datos iniciales:", err);
            taskListBody.html('<tr><td colspan="7" class="text-center text-danger">Error al cargar las tareas.</td></tr>');
        });
    }

    function populateUserDropdown(users) {
        const responsibleSelect = $('#taskResponsible');
        responsibleSelect.empty().append('<option value="">Sin asignar</option>');
        users.forEach(user => {
            responsibleSelect.append(`<option value="${user.nombreCompleto}">${user.nombreCompleto}</option>`);
        });
    }

    function populateProjectDropdown(projects) {
        const projectSelect = $('#taskProject');
        projectSelect.empty().append('<option value="">Seleccione un proyecto</option>');
        projects.forEach(project => {
            projectSelect.append(`<option value="${project.id}">${project.nombre}</option>`);
        });
    }

    function populateParentTaskDropdown(projectId, currentTaskId = null) {
        const parentTaskSelect = $('#taskParent');
        parentTaskSelect.empty().append('<option value="">Ninguna (Tarea Principal)</option>');
        const potentialParents = allTasks.filter(task => task.project_id === projectId && task.id !== currentTaskId);
        potentialParents.forEach(task => {
            parentTaskSelect.append(`<option value="${task.id}">${task.name}</option>`);
        });
    }

    function initializeGantt(tasks) {
        ganttElement.empty();
        if (!tasks || tasks.length === 0) {
            ganttElement.html('<div class="alert alert-info">No hay tareas para mostrar en el Gantt.</div>');
            return;
        }
        gantt = new Gantt("#gantt", tasks, {
            header_height: 50, column_width: 30, step: 24,
            view_modes: ['Day', 'Week', 'Month'],
            bar_height: 20, padding: 18, view_mode: 'Week', language: 'es',
            on_click: function (task) {
                const originalTask = allTasks.find(t => t.id === task.id);
                if(originalTask) showTaskModal(originalTask);
            }
        });
    }

    function filterTasks() {
        const filterTaskName = $('#filterTaskName').val().toLowerCase();
        const filterTaskStart = $('#filterTaskStart').val();
        const filterTaskEnd = $('#filterTaskEnd').val();
        const filterTaskProgress = $('#filterTaskProgress').val().trim();
        const filterTaskResponsible = $('#filterTaskResponsibleText').val().toLowerCase();
        const filterProject = $('#filterTaskProject').val().toLowerCase();

        const filteredTasks = allTasks.filter(task => {
            const projectName = (allProjects.find(p => p.id === task.project_id) || {}).nombre || '';
            if (filterProject && !projectName.toLowerCase().includes(filterProject)) return false;
            const taskName = String(task.name || '').toLowerCase();
            if (filterTaskName && !taskName.includes(filterTaskName)) return false;
            const taskResponsible = String(task.responsible || '').toLowerCase();
            if (filterTaskResponsible && !taskResponsible.includes(filterTaskResponsible)) return false;
            if (filterTaskStart && task.end < filterTaskStart) return false;
            if (filterTaskEnd && task.start > filterTaskEnd) return false;
            if (filterTaskProgress !== '') {
                let operator = filterTaskProgress.charAt(0);
                let value;
                if (operator === '>' || operator === '<') {
                    value = parseInt(filterTaskProgress.substring(1), 10);
                    if (!isNaN(value)) {
                        if (operator === '>' && parseInt(task.progress) <= value) return false;
                        if (operator === '<' && parseInt(task.progress) >= value) return false;
                    }
                } else {
                    value = parseInt(filterTaskProgress, 10);
                    if (!isNaN(value) && parseInt(task.progress) !== value) return false;
                }
            }
            return true;
        });

        renderTaskList(filteredTasks);
        const tasksForGantt = JSON.parse(JSON.stringify(filteredTasks));
        initializeGantt(tasksForGantt);
    }

    function setupEventListeners() {
        $('[data-bs-target="#taskModal"]').off('click').on('click', function() {
            showTaskModal(null);
        });

        $('#taskProject').on('change', function() {
            const selectedProjectId = $(this).val();
            const currentTaskId = $('#taskId').val();
            populateParentTaskDropdown(selectedProjectId, currentTaskId);
        });

        taskForm.off('submit').on('submit', function(e) {
            e.preventDefault();
            const taskId = $('#taskId').val();
            const projectId = $('#taskProject').val();
            if (!projectId) { alert('Debe seleccionar un proyecto para la tarea.'); return; }

            const taskData = {
                id: taskId || 'Task_' + Date.now(),
                project_id: projectId,
                parent_id: $('#taskParent').val() || null,
                name: $('#taskName').val(),
                start: $('#taskStart').val(),
                end: $('#taskEnd').val(),
                progress: parseInt($('#taskProgress').val(), 10) || 0,
                responsible: $('#taskResponsible').val(),
                dependencies: $('#taskDependencies').val(),
                trabajos_realizados: currentTaskWorkLogs
            };

            const isEditing = !!taskId;
            $.ajax({
                url: isEditing ? `${tasksApiUrl}?id=${taskId}` : tasksApiUrl,
                type: isEditing ? 'PUT' : 'POST',
                contentType: 'application/json',
                data: JSON.stringify(taskData),
                success: function() { taskModal.hide(); init(); },
                error: function() { alert('Error al guardar la tarea.'); }
            });
        });

        $('#btnDeleteTask').off('click').on('click', function() {
            const taskId = $('#taskId').val();
            if (taskId && confirm('¿Estás seguro de que quieres eliminar esta tarea? Las subtareas también serán eliminadas.')) {
                $.ajax({
                    url: `${tasksApiUrl}?id=${taskId}`,
                    type: 'DELETE',
                    success: function() { taskModal.hide(); init(); },
                    error: function() { alert('Error al eliminar la tarea.'); }
                });
            }
        });

        $('#addCommentBtn').off('click').on('click', function() {
            const taskId = $('#taskId').val();
            const commentText = $('#newCommentText').val().trim();
            const attachmentFile = $('#commentAttachment')[0].files[0];

            if (!taskId || !commentText) {
                alert('No se puede enviar un comentario vacío.');
                return;
            }

            const formData = new FormData();
            formData.append('task_id', taskId);
            formData.append('comment_text', commentText);
            if (attachmentFile) {
                formData.append('attachment', attachmentFile);
            }

            $.ajax({
                url: commentsApiUrl,
                type: 'POST',
                data: formData,
                processData: false, // Important!
                contentType: false, // Important!
                success: function() {
                    $('#newCommentText').val('');
                    $('#commentAttachment').val(''); // Clear file input
                    loadAndRenderComments(taskId);
                },
                error: function() {
                    alert('Error al enviar el comentario.');
                }
            });
        });

        // --- @mention Autocomplete Logic ---
        const newCommentText = $('#newCommentText');
        const mentionSuggestions = $('#mention-suggestions');

        newCommentText.on('keyup', function(e) {
            const text = newCommentText.val();
            const cursorPos = newCommentText[0].selectionStart;
            const textBeforeCursor = text.substring(0, cursorPos);
            const mentionMatch = textBeforeCursor.match(/@(\w*)$/);

            if (mentionMatch) {
                const searchTerm = mentionMatch[1].toLowerCase();
                const matchedUsers = allUsers.filter(user => 
                    user.nombreCompleto.toLowerCase().includes(searchTerm)
                );

                if (matchedUsers.length > 0) {
                    renderMentionSuggestions(matchedUsers);
                } else {
                    mentionSuggestions.hide();
                }
            } else {
                mentionSuggestions.hide();
            }
        });

        function renderMentionSuggestions(users) {
            mentionSuggestions.empty().show();
            users.forEach(user => {
                const item = $(`<a class="dropdown-item" href="#">${user.nombreCompleto}</a>`);
                item.on('mousedown', function(e) { // Use mousedown to fire before blur
                    e.preventDefault();
                    const text = newCommentText.val();
                    const cursorPos = newCommentText[0].selectionStart;
                    const textBeforeCursor = text.substring(0, cursorPos);
                    
                    const mentionMatch = textBeforeCursor.match(/@(\w*)$/);
                    if (mentionMatch) {
                        const startIndex = mentionMatch.index;
                        const newText = text.substring(0, startIndex) + `@${user.nombreCompleto} ` + text.substring(cursorPos);
                        newCommentText.val(newText);
                        
                        const newCursorPos = startIndex + user.nombreCompleto.length + 2;
                        newCommentText[0].setSelectionRange(newCursorPos, newCursorPos);
                    }
                    mentionSuggestions.hide();
                    newCommentText.focus();
                });
                mentionSuggestions.append(item);
            });
        }

        // Hide suggestions when textarea loses focus
        newCommentText.on('blur', function() {
            setTimeout(() => mentionSuggestions.hide(), 200); // Delay to allow click
        });

        taskListBody.on('click', '.edit-task-btn', function() {
            const taskId = $(this).closest('.task-row').data('task-id');
            const taskToEdit = allTasks.find(t => t.id === taskId);
            if (taskToEdit) showTaskModal(taskToEdit);
        });

        $('#addWorkLogBtn').off('click').on('click', function() {
            const workLogId = $(this).data('edit-id');
            const workLogType = $('#workLogType').val();
            const workLogDate = $('#workLogDate').val();
            const workLogStartTime = $('#workLogStartTime').val();
            const workLogEndTime = $('#workLogEndTime').val();
            const workLogDescription = $('#workLogDescription').val();
            const workLogLink = $('#workLogLink').val();
            if (!workLogType || !workLogDate || !workLogStartTime || !workLogEndTime) { alert('Por favor, completa el tipo, fecha, hora de inicio y hora de fin del trabajo.'); return; }
            const hours = calculateHours(workLogStartTime, workLogEndTime);
            const newWorkLog = { id: workLogId || 'WorkLog_' + Date.now(), type: workLogType, date: workLogDate, start_time: workLogStartTime, end_time: workLogEndTime, hours: hours, description: workLogDescription, link: workLogLink };
            if (workLogId) {
                const index = currentTaskWorkLogs.findIndex(log => log.id === workLogId);
                if (index !== -1) currentTaskWorkLogs[index] = newWorkLog;
                $(this).removeData('edit-id').text('Agregar Trabajo');
            } else {
                currentTaskWorkLogs.push(newWorkLog);
            }
            renderWorkLogs(currentTaskWorkLogs);
            $('#workLogType, #workLogDate, #workLogStartTime, #workLogEndTime, #workLogDescription, #workLogLink').val('');
        });

        $('#workLogsList').on('click', '.edit-work-log', function() {
            const logId = $(this).data('id');
            const logToEdit = currentTaskWorkLogs.find(log => log.id === logId);
            if (logToEdit) {
                $('#workLogType').val(logToEdit.type);
                $('#workLogDate').val(logToEdit.date);
                $('#workLogStartTime').val(logToEdit.start_time);
                $('#workLogEndTime').val(logToEdit.end_time);
                $('#workLogDescription').val(logToEdit.description);
                $('#workLogLink').val(logToEdit.link);
                $('#addWorkLogBtn').text('Actualizar Trabajo').data('edit-id', logId);
            }
        });

        $('#workLogsList').on('click', '.delete-work-log', function() {
            const logId = $(this).data('id');
            if (confirm('¿Estás seguro de que quieres eliminar este trabajo realizado?')) {
                currentTaskWorkLogs = currentTaskWorkLogs.filter(log => log.id !== logId);
                renderWorkLogs(currentTaskWorkLogs);
            }
        });

        $('.task-filter').on('input change', filterTasks);
        $('#clearFilters').on('click', function() {
            $('.task-filter').val('');
            ganttTitle.text('Gantt de Todos los Proyectos');
            filterTasks();
        });
    }

    function showTaskModal(task) {
        taskForm[0].reset();
        currentTaskWorkLogs = [];
        $('#commentsList').empty();
        $('#addWorkLogBtn').text('Agregar Trabajo').removeData('edit-id');
        $('#task-details-tab').tab('show');

        const progressInput = $('#taskProgress');
        progressInput.prop('disabled', false);

        // Disable fields for team members
        const isTeamMember = currentUserRole === 'team_member';
        $('#taskProject, #taskParent, #taskResponsible, #taskDependencies').prop('disabled', isTeamMember);

        if (task) {
            const taskNode = tasksById[task.id];
            if (taskNode && taskNode.children.length > 0) {
                progressInput.prop('disabled', true);
            }

            $('#taskModalLabel').text('Editar Tarea');
            $('#taskId').val(task.id);
            $('#taskName').val(task.name);
            $('#taskStart').val(task.start);
            $('#taskEnd').val(task.end);
            progressInput.val(task.progress);
            $('#taskResponsible').val(task.responsible);
            $('#taskDependencies').val(task.dependencies);
            $('#taskProject').val(task.project_id);

            // A team member can only edit their own tasks
            if (isTeamMember && task.responsible !== $('#currentUserName').val()) {
                taskModal.hide(); alert('No tienes permiso para editar esta tarea.'); return;
            }
            $('#btnDeleteTask').show();

            populateParentTaskDropdown(task.project_id, task.id);
            $('#taskParent').val(task.parent_id);

            if (task.trabajos_realizados && Array.isArray(task.trabajos_realizados)) {
                currentTaskWorkLogs = [...task.trabajos_realizados];
            }
            // Cargar comentarios
            loadAndRenderComments(task.id);
        } else {
            $('#taskModalLabel').text('Añadir Tarea');
            $('#taskId').val('');
            $('#taskProject').prop('disabled', isTeamMember);
            $('#taskParent').empty().append('<option value="">Seleccione un proyecto primero</option>');
            $('#btnDeleteTask').hide();
        }
        renderWorkLogs(currentTaskWorkLogs);
        taskModal.show();
    }

    function loadAndRenderComments(taskId) {
        const commentsList = $('#commentsList');
        commentsList.html('<p class="text-muted text-center">Cargando comentarios...</p>');

        $.get(`${commentsApiUrl}?task_id=${taskId}`, function(comments) {
            commentsList.empty();
            if (!comments || comments.length === 0) {
                commentsList.append('<p class="text-muted text-center">No hay comentarios para esta tarea.</p>');
                return;
            }

            comments.forEach(comment => {
                const highlightedComment = comment.comment_text.replace(/@([\p{L}\p{N}\s_.-]+?)\b/gu, '<strong>@$1</strong>');
                const attachmentLink = comment.attachment ? `<div class="mt-2"><a href="${comment.attachment}" target="_blank" class="btn btn-sm btn-outline-secondary"><i class="bi bi-paperclip"></i> Ver Adjunto</a></div>` : '';
                const commentHtml = `
                    <div class="list-group-item list-group-item-action flex-column align-items-start mb-2">
                        <div class="d-flex w-100 justify-content-between">
                            <h6 class="mb-1">${comment.user_name}</h6>
                            <small class="text-muted">${formatDateTimeToDDMMYYYYHHMM(comment.timestamp)}</small>
                        </div>
                        <p class="mb-1">${highlightedComment}</p>
                        ${attachmentLink}
                    </div>`;
                commentsList.append(commentHtml);
            });
        }).fail(() => commentsList.html('<p class="text-danger text-center">Error al cargar comentarios.</p>'));
    }

    function renderTaskList(tasks) {
        taskListBody.empty();

        if (!tasks || tasks.length === 0) {
            taskListBody.append('<tr><td colspan="7" class="text-center text-muted">No se encontraron tareas.</td></tr>');
            return;
        }

        const tasksByIdForRender = tasks.reduce((acc, task) => {
            acc[task.id] = { ...task, children: [] };
            return acc;
        }, {});

        const tree = [];
        Object.values(tasksByIdForRender).forEach(task => {
            if (task.parent_id && tasksByIdForRender[task.parent_id]) {
                tasksByIdForRender[task.parent_id].children.push(task);
            } else {
                tree.push(task);
            }
        });

        function renderTaskRowRecursive(task, level) {
            const project = allProjects.find(p => p.id === task.project_id);
            const projectName = project ? project.nombre : 'N/A';
            const indentation = '&nbsp;'.repeat(level * 4);

            const row = `
                <tr class="task-row" data-task-id="${task.id}">
                    <td>${indentation}${task.name}</td>
                    <td>${projectName}</td>
                    <td>${formatDateToDDMMYYYY(task.start)}</td>
                    <td>${formatDateToDDMMYYYY(task.end)}</td>
                    <td>${task.progress}%</td>
                    <td>${task.responsible || '-'}</td>
                    <td>
                        <button type="button" class="btn btn-sm btn-info edit-task-btn">Editar</button>
                    </td>
                </tr>
            `;
            taskListBody.append(row);

            if (task.children.length > 0) {
                task.children.sort((a, b) => new Date(a.start) - new Date(b.start)).forEach(child => renderTaskRowRecursive(child, level + 1));
            }
        }

        tree.sort((a, b) => new Date(a.start) - new Date(b.start)).forEach(task => renderTaskRowRecursive(task, 0));
    }

    init();
});