<?php
// Este script está diseñado para ser ejecutado por una tarea programada (cron job) una vez al día.

// --- Configuración ---
const REMINDER_DAYS = 2; // Enviar recordatorio para tareas que vencen en 2 días o menos.
const FROM_EMAIL = 'no-reply@dunamisproject.com';
const PROJECT_URL = 'http://localhost/dunamisproject'; // Cambia esto a la URL real de tu proyecto.

// --- Setup ---
// Establecer la zona horaria para evitar problemas con fechas y horas.
date_default_timezone_set('America/Asuncion'); // Ajusta a la zona horaria de tu servidor.

echo "Iniciando script de recordatorios de tareas...\n";

// --- Cargar Datos ---
$tasksPath = __DIR__ . '/../data/tasks.json';
$usersPath = __DIR__ . '/../data/usuarios.json';

if (!file_exists($tasksPath) || !file_exists($usersPath)) {
    die("Error: No se encontraron los archivos de datos (tasks.json o usuarios.json).\n");
}

$allTasks = json_decode(file_get_contents($tasksPath), true);
$allUsers = json_decode(file_get_contents($usersPath), true);

if (json_last_error() !== JSON_ERROR_NONE) {
    die("Error: Formato JSON inválido en los archivos de datos.\n");
}

// Crear un mapa de nombres de usuario a correos electrónicos para una búsqueda eficiente.
$userEmailMap = [];
foreach ($allUsers as $user) {
    if (isset($user['nombreCompleto']) && isset($user['email'])) {
        $userEmailMap[$user['nombreCompleto']] = $user['email'];
    }
}

// --- Lógica del Script ---
$today = new DateTime();
$today->setTime(0, 0, 0); // Ignorar la hora para comparar solo fechas.

$remindersSent = 0;

foreach ($allTasks as $task) {
    // Omitir tareas ya completadas o sin fecha de finalización.
    if (empty($task['end']) || (isset($task['progress']) && (int)$task['progress'] === 100)) {
        continue;
    }

    try {
        $endDate = new DateTime($task['end']);
        $endDate->setTime(0, 0, 0);
    } catch (Exception $e) {
        // Omitir si la fecha es inválida.
        continue;
    }

    // Calcular la diferencia en días.
    $interval = $today->diff($endDate);
    $daysRemaining = (int)$interval->format('%r%a'); // %r da el signo (+ o -)

    // Comprobar si la tarea vence dentro del umbral de recordatorio.
    if ($daysRemaining >= 0 && $daysRemaining <= REMINDER_DAYS) {
        $responsibleName = $task['responsible'] ?? null;
        if (!$responsibleName) {
            continue; // No se puede enviar recordatorio si no hay responsable.
        }

        $recipientEmail = $userEmailMap[$responsibleName] ?? null;
        if (!$recipientEmail) {
            echo "Advertencia: No se encontró el email para el usuario '{$responsibleName}'.\n";
            continue;
        }

        // --- Enviar Correo Electrónico ---
        $subject = "Recordatorio: La tarea '{$task['name']}' vence pronto";
        $taskUrl = PROJECT_URL . "/tareas.php?project_id=" . ($task['project_id'] ?? '');
        $message = "Hola {$responsibleName},\n\n";
        $message .= "Este es un recordatorio de que la tarea '{$task['name']}' está a punto de vencer.\n\n";
        $message .= "Fecha de Vencimiento: " . $endDate->format('d/m/Y') . " (en {$daysRemaining} día(s)).\n";
        $message .= "Progreso Actual: " . ($task['progress'] ?? 0) . "%\n\n";
        $message .= "Puedes ver los detalles de la tarea aquí: {$taskUrl}\n\n";
        $message .= "Saludos,\nEl equipo de Dunamis Project";

        $headers = 'From: ' . FROM_EMAIL;

        // La función mail() de PHP se usa aquí.
        if (mail($recipientEmail, $subject, $message, $headers)) {
            echo "Recordatorio enviado a {$recipientEmail} para la tarea '{$task['name']}'.\n";
            $remindersSent++;
        } else {
            echo "Error: No se pudo enviar el correo a {$recipientEmail}.\n";
        }
    }
}

echo "Script finalizado. Se enviaron {$remindersSent} recordatorios.\n";
?>