$(document).ready(function() {
    const projectId = $('#projectId').val();
    const projectNameSpan = $('#projectName');
    const financialBudget = $('#financial-budget');
    const financialTotalCost = $('#financial-total-cost');
    const financialBalance = $('#financial-balance');
    const costWorkLogs = $('#cost-work-logs');
    const costExpenses = $('#cost-expenses');
    const expensesTableBody = $('#expensesTableBody');
    const expenseForm = $('#expenseForm');
    const expenseId = $('#expenseId');
    const expenseDescription = $('#expenseDescription');
    const expenseAmount = $('#expenseAmount');
    const expenseDate = $('#expenseDate');

    const projectsApiUrl = 'api/projects.php';
    const financialsApiUrl = 'api/financials.php';
    const expensesApiUrl = 'api/expenses.php';

    function formatNumberForDisplay(amount) {
        const formatted = Number(amount).toLocaleString('es-ES', { minimumFractionDigits: 2, maximumFractionDigits: 2 });
        console.log('Formatted amount:', formatted);
        return formatted;
    }

    function parseFormattedNumber(str) {
        // Remove thousands separator (dots) and replace decimal comma with a dot
        return parseFloat(str.replace(/\./g, '').replace(',', '.'));
    }

    function loadProjectDetails() {
        $.get(`${projectsApiUrl}?id=${projectId}`, function(projects) {
            if (projects.length > 0) {
                projectNameSpan.text(projects[0].nombre);
            }
        });
    }

    function loadFinancialSummary() {
        $.get(`${financialsApiUrl}?project_id=${projectId}`, function(data) {
            console.log('Budget:', typeof data.budget, data.budget);
            console.log('Total Cost:', typeof data.total_cost, data.total_cost);
            console.log('Balance:', typeof data.balance, data.balance);
            console.log('Work Log Cost:', typeof data.total_work_log_cost, data.total_work_log_cost);
            console.log('Total Expenses:', typeof data.total_expenses, data.total_expenses);
            financialBudget.text(formatNumberForDisplay(data.budget));
            financialTotalCost.text(formatNumberForDisplay(data.total_cost));
            financialBalance.text(formatNumberForDisplay(data.balance));
            costWorkLogs.text(formatNumberForDisplay(data.total_work_log_cost));
            costExpenses.text(formatNumberForDisplay(data.total_expenses));
        });
    }

    function loadExpenses() {
        $.get(`${expensesApiUrl}?project_id=${projectId}`, function(expenses) {
            expensesTableBody.empty();
            if (expenses.length === 0) {
                expensesTableBody.append('<tr><td colspan="4" class="text-center">No hay gastos registrados.</td></tr>');
            } else {
                expenses.forEach(expense => {
                    expensesTableBody.append(`
                        <tr data-id="${expense.id}">
                            <td>${expense.description}</td>
                            <td>${formatNumberForDisplay(expense.amount)}</td>
                            <td>${expense.date}</td>
                            <td>
                                <button class="btn btn-sm btn-info btn-edit-expense">Editar</button>
                                <button class="btn btn-sm btn-danger btn-delete-expense">Eliminar</button>
                            </td>
                        </tr>
                    `);
                });
            }
        });
    }

    expenseForm.submit(function(e) {
        e.preventDefault();
        const expenseData = {
            project_id: projectId,
            description: expenseDescription.val(),
            amount: parseFormattedNumber(expenseAmount.val()),
            date: expenseDate.val()
        };

        const isEditing = !!expenseId.val();
        const url = isEditing ? `${expensesApiUrl}?id=${expenseId.val()}` : expensesApiUrl;
        const type = isEditing ? 'PUT' : 'POST';

        $.ajax({
            url: url,
            type: type,
            contentType: 'application/json',
            data: JSON.stringify(expenseData),
            success: function() {
                expenseForm[0].reset();
                expenseId.val('');
                loadFinancialSummary();
                loadExpenses();
            },
            error: function() {
                alert('Error al guardar el gasto.');
            }
        });
    });

    expensesTableBody.on('click', '.btn-edit-expense', function() {
        const id = $(this).closest('tr').data('id');
        $.get(`${expensesApiUrl}?id=${id}`, function(expenses) {
            const expense = expenses.find(e => e.id === id);
            if (expense) {
                expenseId.val(expense.id);
                expenseDescription.val(expense.description);
                expenseAmount.val(formatNumberForDisplay(expense.amount));
                expenseDate.val(expense.date);
            }
        });
    });

    expensesTableBody.on('click', '.btn-delete-expense', function() {
        const id = $(this).closest('tr').data('id');
        if (confirm('¿Estás seguro de que quieres eliminar este gasto?')) {
            $.ajax({
                url: `${expensesApiUrl}?id=${id}`,
                type: 'DELETE',
                success: function() {
                    loadFinancialSummary();
                    loadExpenses();
                },
                error: function() {
                    alert('Error al eliminar el gasto.');
                }
            });
        }
    });

    loadProjectDetails();
    loadFinancialSummary();
    loadExpenses();
});