function getDotColorClass(daysRemaining) {
    if (daysRemaining === null) {
        return 'bg-secondary'; // Or any other color for N/A
    } else if (daysRemaining <= 0) {
        return 'bg-danger'; // Red for overdue/due today
    } else if (daysRemaining <= 2) {
        return 'bg-warning'; // Yellow for expiring soon
    } else {
        return 'bg-success'; // Green for plenty of time
    }
}

function formatNumber(value) {
    if (typeof value !== 'number') {
        value = parseFloat(value);
    }
    if (isNaN(value)) {
        return 'N/A';
    }
    return value.toLocaleString('es-PY', { minimumFractionDigits: 0, maximumFractionDigits: 2 });
}

function formatTimeAgo(timestamp) {
    const now = new Date();
    const past = new Date(timestamp);
    const seconds = Math.floor((now - past) / 1000);

    let interval = seconds / 31536000;
    if (interval > 1) return `hace ${Math.floor(interval)} años`;
    interval = seconds / 2592000;
    if (interval > 1) return `hace ${Math.floor(interval)} meses`;
    interval = seconds / 86400;
    if (interval > 1) return `hace ${Math.floor(interval)} días`;
    interval = seconds / 3600;
    if (interval > 1) return `hace ${Math.floor(interval)} horas`;
    interval = seconds / 60;
    if (interval > 1) return `hace ${Math.floor(interval)} minutos`;
    return `hace unos segundos`;
}

$(document).ready(function() {
    const userEmail = $('#userEmail').val();

    function fetchAndRenderProjects() {
        $.ajax({
            url: 'api/projects.php',
            method: 'GET',
            data: { user_email: userEmail },
            success: function(projects) {
                const projectsList = $('#my-projects-list');
                projectsList.empty();
                if (projects.length > 0) {
                    projects.forEach(project => {
                        const projectDotClass = getDotColorClass(project.days_remaining);
                        projectsList.append(`
                            <a href="tareas.php?project_id=${project.id}" class="card mb-2 text-decoration-none text-dark">
                                <div class="card-body">
                                    <h5 class="card-title d-flex justify-content-between align-items-center">
                                        ${project.nombre}
                                        <span class="badge ${projectDotClass} rounded-circle p-2"></span>
                                    </h5>
                                    <p class="card-text">Cliente: ${project.client_name}</p>
                                    <p class="card-text">Estado: ${project.estado}</p>
                                    <p class="card-text">Progreso General: ${formatNumber(project.progress_general)}%</p>
                                    <p class="card-text">Días Restantes: ${project.days_remaining !== null ? project.days_remaining : 'N/A'}</p>
                                </div>
                            </a>
                        `);
                    });
                } else {
                    projectsList.append('<p class="text-muted">No hay proyectos asignados a usted.</p>');
                }
            },
            error: function(xhr, status, error) {
                $('#my-projects-list').empty().append('<p class="text-danger">Error al cargar proyectos.</p>');
                console.error("Error fetching projects:", status, error);
            }
        });
    }

    function fetchAndRenderTasks() {
        $.ajax({
            url: 'api/tasks.php',
            method: 'GET',
            data: { user_email: userEmail },
            success: function(tasks) {
                const tasksList = $('#my-tasks-list');
                tasksList.empty();
                if (tasks.length > 0) {
                    tasks.forEach(task => {
                        const taskDotClass = getDotColorClass(task.days_remaining);
                        tasksList.append(`
                            <div class="card mb-2">
                                <div class="card-body">
                                    <h5 class="card-title d-flex justify-content-between align-items-center">
                                        ${task.name}
                                        <span class="badge ${taskDotClass} rounded-circle p-2"></span>
                                    </h5>
                                    <p class="card-text">Proyecto ID: ${task.project_id}</p>
                                    <p class="card-text">Progreso: ${formatNumber(task.progress)}%</p>
                                    <p class="card-text">Días Restantes: ${task.days_remaining !== null ? task.days_remaining : 'N/A'}</p>
                                </div>
                            </div>
                        `);
                    });
                } else {
                    tasksList.append('<p class="text-muted">No hay tareas asignadas a usted.</p>');
                }
            },
            error: function(xhr, status, error) {
                $('#my-tasks-list').empty().append('<p class="text-danger">Error al cargar tareas.</p>');
                console.error("Error fetching tasks:", status, error);
            }
        });
    }

    function fetchAndRenderStats() {
        const getProjectsPromise = $.ajax({ url: 'api/projects.php', method: 'GET' });
        const getTasksPromise = $.ajax({ url: 'api/tasks.php', method: 'GET' });
        const getActivityPromise = $.ajax({ url: 'api/activity_log.php', method: 'GET' });

        Promise.all([getProjectsPromise, getTasksPromise, getActivityPromise])
            .then(function([allProjects, allTasks, allActivities]) {
                // Total Proyectos
                $('#stat-total-projects').text(formatNumber(allProjects.length));

                // Proyectos Activos y Completados
                const activeProjects = allProjects.filter(p => p.estado === 'activos').length;
                const completedProjects = allProjects.filter(p => p.estado === 'completado').length;
                $('#stat-active-projects').text(formatNumber(activeProjects));
                $('#stat-completed-projects').text(formatNumber(completedProjects));

                // Total Tareas
                $('#stat-total-tasks').text(formatNumber(allTasks.length));

                // Tareas Pendientes (progreso < 100%)
                const pendingTasks = allTasks.filter(t => (t.progress || 0) < 100).length;
                $('#stat-pending-tasks').text(formatNumber(pendingTasks));

                // Horas Totales Registradas
                let totalHours = 0;
                allTasks.forEach(task => {
                    if (task.trabajos_realizados && Array.isArray(task.trabajos_realizados)) {
                        task.trabajos_realizados.forEach(workLog => {
                            totalHours += parseFloat(workLog.hours) || 0;
                        });
                    }
                });
                $('#stat-total-hours').text(formatNumber(totalHours)); // Mostrar con 2 decimales

                // Render charts after stats are fetched
                renderProjectsStatusChart(allProjects);
                renderWorkLogTypeChart(allTasks);
                renderRecentActivityChart(allActivities);
                renderActivityFeed(allActivities);
                setupActivityChartFilter(allActivities);
            })
            .catch(function(error) {
                console.error("Error fetching stats:", error);
                $('#general-stats').html('<div class="col-12"><div class="alert alert-danger">Error al cargar las estadísticas.</div></div>');
            });
    }

    function renderActivityFeed(activities) {
        const feedList = $('#activity-feed-list');
        feedList.empty();

        if (!activities || activities.length === 0) {
            feedList.append('<p class="text-muted text-center">No hay actividad reciente.</p>');
            return;
        }

        // Show the latest 15 activities
        activities.slice(0, 15).forEach(activity => {
            const feedItem = `
                <div class="list-group-item">
                    <div class="d-flex w-100 justify-content-between">
                        <p class="mb-1"><strong>${activity.user_name}</strong> ${activity.action}</p>
                        <small class="text-muted">${formatTimeAgo(activity.timestamp)}</small>
                    </div>
                </div>
            `;
            feedList.append(feedItem);
        });
    }

    function renderProjectsStatusChart(projects) {
        const ctx = document.getElementById('projectsStatusChart').getContext('2d');
        const statusCounts = {};
        projects.forEach(project => {
            statusCounts[project.estado] = (statusCounts[project.estado] || 0) + 1;
        });

        const labels = Object.keys(statusCounts);
        const data = Object.values(statusCounts);
        const backgroundColors = [
            'rgba(255, 99, 132, 0.7)', // Red
            'rgba(54, 162, 235, 0.7)', // Blue
            'rgba(255, 206, 86, 0.7)', // Yellow
            'rgba(75, 192, 192, 0.7)', // Green
            'rgba(153, 102, 255, 0.7)', // Purple
            'rgba(255, 159, 64, 0.7)'  // Orange
        ];

        new Chart(ctx, {
            type: 'doughnut',
            data: {
                labels: labels,
                datasets: [{
                    data: data,
                    backgroundColor: backgroundColors.slice(0, labels.length),
                    borderColor: '#fff',
                    borderWidth: 1
                }]
            },
            options: {
                responsive: true,
                plugins: {
                    legend: {
                        position: 'top',
                    },
                    title: {
                        display: true,
                        text: 'Proyectos por Estado'
                    }
                }
            }
        });
    }

    function renderWorkLogTypeChart(tasks) {
        const ctx = document.getElementById('workLogTypeChart').getContext('2d');
        const workLogTypeHours = {};

        tasks.forEach(task => {
            if (task.trabajos_realizados && Array.isArray(task.trabajos_realizados)) {
                task.trabajos_realizados.forEach(workLog => {
                    const type = workLog.type || 'Sin Tipo'; // Assuming a 'type' field in workLog
                    workLogTypeHours[type] = (workLogTypeHours[type] || 0) + (parseFloat(workLog.hours) || 0);
                });
            }
        });

        const labels = Object.keys(workLogTypeHours);
        const data = Object.values(workLogTypeHours);
        const backgroundColors = [
            'rgba(255, 99, 132, 0.7)',
            'rgba(54, 162, 235, 0.7)',
            'rgba(255, 206, 86, 0.7)',
            'rgba(75, 192, 192, 0.7)',
            'rgba(153, 102, 255, 0.7)',
            'rgba(255, 159, 64, 0.7)'
        ];

        new Chart(ctx, {
            type: 'bar', // Bar chart for work log types
            data: {
                labels: labels,
                datasets: [{
                    label: 'Horas Registradas',
                    data: data,
                    backgroundColor: backgroundColors.slice(0, labels.length),
                    borderColor: '#fff',
                    borderWidth: 1
                }]
            },
            options: {
                responsive: true,
                plugins: {
                    legend: {
                        display: false,
                    },
                    title: {
                        display: true,
                        text: 'Horas por Tipo de Trabajo'
                    }
                },
                scales: {
                    y: {
                        beginAtZero: true,
                        title: {
                            display: true,
                            text: 'Horas'
                        }
                    }
                }
            }
        });
    }

    let recentActivityChartInstance = null;

    function renderRecentActivityChart(activities) {
        if (recentActivityChartInstance) {
            recentActivityChartInstance.destroy();
        }

        const ctx = document.getElementById('recentActivityChart').getContext('2d');
        
        let startDateString = $('#activityStartDate').val();
        let endDateString = $('#activityEndDate').val();

        let startDate, endDate;

        if (!startDateString || !endDateString) {
            endDate = new Date();
            startDate = new Date();
            startDate.setDate(endDate.getDate() - 6);

            $('#activityStartDate').val(startDate.toISOString().split('T')[0]);
            $('#activityEndDate').val(endDate.toISOString().split('T')[0]);
        } else {
            startDate = new Date(startDateString);
            endDate = new Date(endDateString);
        }

        if (startDate > endDate) {
            [startDate, endDate] = [endDate, startDate]; // Swap dates
        }

        startDate.setHours(0, 0, 0, 0);
        endDate.setHours(23, 59, 59, 999);

        const activityByDay = {};
        const labels = [];
        
        for (let d = new Date(startDate); d <= endDate; d.setDate(d.getDate() + 1)) {
            const date = new Date(d);
            const dayString = date.toISOString().split('T')[0];
            labels.push(`${date.getDate()}/${date.getMonth() + 1}`);
            activityByDay[dayString] = 0;
        }

        const filteredActivities = activities.filter(activity => {
            const activityDate = new Date(activity.timestamp);
            return activityDate >= startDate && activityDate <= endDate;
        });

        filteredActivities.forEach(activity => {
            const dayString = new Date(activity.timestamp).toISOString().split('T')[0];
            if (activityByDay.hasOwnProperty(dayString)) {
                activityByDay[dayString]++;
            }
        });

        const data = Object.values(activityByDay);

        recentActivityChartInstance = new Chart(ctx, {
            type: 'bar',
            data: {
                labels: labels,
                datasets: [{
                    label: 'Nº de Actividades',
                    data: data,
                    backgroundColor: 'rgba(75, 192, 192, 0.7)',
                    borderColor: 'rgba(75, 192, 192, 1)',
                    borderWidth: 1
                }]
            },
            options: {
                responsive: true,
                plugins: {
                    legend: {
                        display: false
                    }
                },
                scales: {
                    y: {
                        beginAtZero: true,
                        ticks: {
                            stepSize: 1 // Ensure y-axis shows whole numbers
                        }
                    }
                }
            }
        });
    }

    function setupActivityChartFilter(allActivities) {
        $('#activityStartDate, #activityEndDate').on('change', function() {
            renderRecentActivityChart(allActivities);
        });
    }

    fetchAndRenderProjects();
    fetchAndRenderTasks();
    fetchAndRenderStats();
});