$(document).ready(function() {
    const projectsApiUrl = 'api/projects.php';
    const usersApiUrl = 'api/usuarios.php';
    const clientsApiUrl = 'api/clients.php';
    const activityLogApiUrl = 'api/activity_log.php';

    function formatNumberForDisplay(amount) {
        return Number(amount).toLocaleString('es-PY', { minimumFractionDigits: 0, maximumFractionDigits: 0 });
    }

    function parseFormattedNumber(str) {
        return parseFloat(str.replace(/\./g, '').replace(',', '.'));
    }
    
    const projectListContainer = $('#project-list');
    const projectModal = new bootstrap.Modal(document.getElementById('projectModal'));
    const projectForm = $('#projectForm');

    let allProjects = [];
    let allUsers = [];
    let allClients = [];

    function init() {
        const getProjects = $.get(projectsApiUrl);
        const getUsers = $.get(usersApiUrl);
        const getClients = $.get(clientsApiUrl);

        Promise.all([getProjects, getUsers, getClients]).then(function([projects, users, clients]) {
            allProjects = projects;
            allUsers = users;
            allClients = clients;

            populateUserDropdowns();
            populateClientDropdown();
            renderProjects();
            setupEventListeners();
        }).fail(function() {
            projectListContainer.html('<div class="col"><div class="alert alert-danger">Error al cargar los datos iniciales.</div></div>');
        });
    }

    function populateUserDropdowns() {
        const managerSelect = $('#projectManager');
        const sponsorSelect = $('#projectSponsor');
        managerSelect.empty().append('<option value="">Seleccionar...</option>');
        sponsorSelect.empty().append('<option value="">Seleccionar...</option>');

        allUsers.forEach(user => {
            const option = `<option value="${user.nombreCompleto}">${user.nombreCompleto}</option>`;
            managerSelect.append(option);
            sponsorSelect.append(option);
        });
    }

    function populateClientDropdown() {
        const clientSelect = $('#projectClient');
        clientSelect.empty().append('<option value="">Seleccionar Cliente...</option>');
        allClients.forEach(client => {
            clientSelect.append(`<option value="${client.id}" data-name="${client.nombre_razon_social}">${client.nombre_razon_social}</option>`);
        });
    }

    function getStatusBadge(status) {
        let badgeClass = 'bg-secondary';
        switch (status) {
            case 'activos': badgeClass = 'bg-success'; break;
            case 'planificacion': badgeClass = 'bg-info text-dark'; break;
            case 'completado': badgeClass = 'bg-primary'; break;
            case 'en espera': badgeClass = 'bg-warning text-dark'; break;
            case 'cancelado': badgeClass = 'bg-danger'; break;
            case 'pendiente de aprobacion': badgeClass = 'bg-light text-dark'; break;
        }
        const formattedStatus = status ? status.charAt(0).toUpperCase() + status.slice(1) : 'Sin estado';
        return `<span class="badge ${badgeClass}">${formattedStatus}</span>`;
    }

    function getDaysRemainingBadge(days) {
        let badgeClass = 'bg-secondary';
        let badgeText = '';

        if (days === null) {
            badgeText = 'Sin fecha de fin';
        } else if (days < 0) {
            badgeClass = 'bg-danger';
            badgeText = `Vencido hace ${Math.abs(days)} días`;
        } else if (days <= 2) {
            badgeClass = 'bg-warning text-dark';
            badgeText = `Vence en ${days} día${days === 1 ? '' : 's'}`;
        } else {
            badgeClass = 'bg-success';
            badgeText = `Vence en ${days} días`;
        }
        return `<span class="badge ${badgeClass}">${badgeText}</span>`;
    }

    function renderProjects() {
        projectListContainer.empty();

        const activeProjects = allProjects.filter(project => project.estado !== 'completado' && project.estado !== 'cancelado');
        const closedProjects = allProjects.filter(project => project.estado === 'completado' || project.estado === 'cancelado');

        // Render Active Projects
        if (activeProjects.length > 0) {
            projectListContainer.append('<h3 class="mt-4 mb-3">Proyectos Activos</h3>');
            const activeProjectsRow = $('<div class="row gy-4 mb-5"></div>');
            activeProjects.forEach(project => {
                const projectCard = `
                    <div class="col-md-6 col-lg-4 d-flex">
                        <div class="card h-100 shadow-sm w-100 project-card" style="cursor: pointer;" data-id="${project.id}">
                            <div class="card-body d-flex flex-column">
                                <h5 class="card-title">${project.nombre}</h5>
                                <p class="card-text text-primary mb-2"><strong>Cliente:</strong> ${project.client_name || 'N/A'}</p>
                                <div class="mb-3">
                                    ${getStatusBadge(project.estado)}
                                    ${getDaysRemainingBadge(project.days_remaining)}
                                </div>
                                <p class="card-text small mb-1"><strong>Gerente:</strong> ${project.gerente_proyecto || 'N/A'}</p>
                                <p class="card-text small"><strong>Sponsor:</strong> ${project.sponsor_proyecto || 'N/A'}</p>
                                <p class="card-text small"><strong>Presupuesto:</strong> ${formatNumberForDisplay(project.budget || 0)}</p>
                                <div class="progress mb-2">
                                    <div class="progress-bar" role="progressbar" style="width: ${project.progress_general || 0}%;" aria-valuenow="${project.progress_general || 0}" aria-valuemin="0" aria-valuemax="100">
                                        ${project.progress_general || 0}%
                                    </div>
                                </div>
                                <div class="mt-auto pt-3">
                                    <a href="tareas.php?project_id=${project.id}" class="btn btn-outline-primary w-100 mb-2">Ver Tareas y Gantt</a>
                                    <a href="project_financials.php?project_id=${project.id}" class="btn btn-outline-info w-100 mb-2">Ver Finanzas</a>
                                    <button class="btn btn-sm btn-outline-secondary btn-edit-project" data-id="${project.id}">Editar</button>
                                </div>
                            </div>
                            <div class="card-footer text-muted small">
                                Inicio: ${project.fecha_inicio} | Fin: ${project.terminos}
                            </div>
                        </div>
                    </div>
                `;
                activeProjectsRow.append(projectCard);
            });
            projectListContainer.append(activeProjectsRow);
        } else {
            projectListContainer.append('<h3 class="mt-4 mb-3">Proyectos Activos</h3><div class="alert alert-info">No hay proyectos activos para mostrar.</div>');
        }

        // Render Closed Projects
        if (closedProjects.length > 0) {
            projectListContainer.append('<h3 class="mt-4 mb-3">Proyectos Cerrados</h3>');
            const closedProjectsRow = $('<div class="row gy-4"></div>');
            closedProjects.forEach(project => {
                const projectCard = `
                    <div class="col-md-6 col-lg-4 d-flex">
                        <div class="card h-100 shadow-sm w-100 project-card" style="cursor: pointer;" data-id="${project.id}">
                            <div class="card-body d-flex flex-column">
                                <h5 class="card-title">${project.nombre}</h5>
                                <p class="card-text text-primary mb-2"><strong>Cliente:</strong> ${project.client_name || 'N/A'}</p>
                                <div class="mb-3">
                                    ${getStatusBadge(project.estado)}
                                    ${getDaysRemainingBadge(project.days_remaining)}
                                </div>
                                <p class="card-text small mb-1"><strong>Gerente:</strong> ${project.gerente_proyecto || 'N/A'}</p>
                                <p class="card-text small"><strong>Sponsor:</strong> ${project.sponsor_proyecto || 'N/A'}</p>
                                <p class="card-text small"><strong>Presupuesto:</strong> ${formatNumberForDisplay(project.budget || 0)}</p>
                                <div class="progress mb-2">
                                    <div class="progress-bar" role="progressbar" style="width: ${project.progress_general || 0}%;" aria-valuenow="${project.progress_general || 0}" aria-valuemin="0" aria-valuemax="100">
                                        ${project.progress_general || 0}%
                                    </div>
                                </div>
                                <div class="mt-auto pt-3">
                                    <a href="tareas.php?project_id=${project.id}" class="btn btn-outline-primary w-100 mb-2">Ver Tareas y Gantt</a>
                                    <a href="project_financials.php?project_id=${project.id}" class="btn btn-outline-info w-100 mb-2">Ver Finanzas</a>
                                    <button class="btn btn-sm btn-outline-secondary btn-edit-project" data-id="${project.id}">Editar</button>
                                </div>
                            </div>
                            <div class="card-footer text-muted small">
                                Inicio: ${project.fecha_inicio} | Fin: ${project.terminos}
                            </div>
                        </div>
                    </div>
                `;
                closedProjectsRow.append(projectCard);
            });
            projectListContainer.append(closedProjectsRow);
        } else if (activeProjects.length === 0) { // Only show "No projects" if both are empty
            projectListContainer.html('<div class="col"><div class="alert alert-info">No hay proyectos para mostrar. ¡Crea uno para comenzar!</div></div>');
        }
    }

    function setupEventListeners() {
        // "Nuevo Proyecto" button
        $('[data-bs-target="#projectModal"]').off('click').on('click', function() {
            showProjectModal(null);
        });

        // Form submission
        projectForm.off('submit').on('submit', function(e) {
            e.preventDefault();
            const projectId = $('#projectId').val();
            const selectedClient = $('#projectClient option:selected');
            
            const projectData = {
                nombre: $('#projectName').val(),
                fecha_inicio: $('#projectStart').val(),
                terminos: $('#projectEnd').val(),
                gerente_proyecto: $('#projectManager').val(),
                sponsor_proyecto: $('#projectSponsor').val(),
                client_id: selectedClient.val(),
                client_name: selectedClient.data('name') || '',
                objetivos: $('#projectObjectives').val(),
                entregables: $('#projectDeliverables').val(),
                estado: $('#projectStatus').val(),                
                budget: parseFormattedNumber($('#projectBudget').val()) || 0,
            };

            const isEditing = !!projectId;
            $.ajax({
                url: isEditing ? `${projectsApiUrl}?id=${projectId}` : projectsApiUrl,
                type: isEditing ? 'PUT' : 'POST',
                contentType: 'application/json',
                data: JSON.stringify(projectData),
                success: function() {
                    projectModal.hide();
                    init();
                },
                error: function() { alert('Error al guardar el proyecto.'); }
            });
        });

        // Delegated event for Edit buttons
        projectListContainer.off('click', '.btn-edit-project').on('click', '.btn-edit-project', function() {
            const projectId = $(this).data('id');
            const project = allProjects.find(p => p.id === projectId);
            if (project) {
                showProjectModal(project);
            }
        });

        // Delegated event for Double-click on project card
        projectListContainer.off('dblclick', '.project-card').on('dblclick', '.project-card', function() {
            const projectId = $(this).data('id');
            if (projectId) {
                window.location.href = `tareas.php?project_id=${projectId}`;
            }
        });
        
        // Delete button inside modal
        $('#btnDeleteProject').off('click').on('click', function() {
            const projectId = $('#projectId').val();
            if (projectId && confirm('¿Estás seguro de que quieres eliminar este proyecto? Sus tareas no serán eliminadas.')) {
                $.ajax({
                    url: `${projectsApiUrl}?id=${projectId}`,
                    type: 'DELETE',
                    success: function() {
                        projectModal.hide();
                        init();
                    },
                    error: function() { alert('Error al eliminar el proyecto.'); }
                });
            }
        });
    }

    function formatDateTime(dateTimeString) {
        if (!dateTimeString) return 'N/A';
        const date = new Date(dateTimeString);
        const day = String(date.getDate()).padStart(2, '0');
        const month = String(date.getMonth() + 1).padStart(2, '0');
        const year = date.getFullYear();
        const hours = String(date.getHours()).padStart(2, '0');
        const minutes = String(date.getMinutes()).padStart(2, '0');
        return `${day}/${month}/${year} ${hours}:${minutes}`;
    }

    function showProjectModal(project) {
        projectForm[0].reset();
        $('#details-tab').tab('show');
        $('#activityLogList').empty().html('<p class="text-muted text-center">Cargando...</p>');

        if (project) { // Edit mode
            $('#projectModalLabel').text('Editar Proyecto');
            $('#projectId').val(project.id);
            $('#projectName').val(project.nombre);
            $('#projectStart').val(project.fecha_inicio);
            $('#projectEnd').val(project.terminos);
            $('#projectManager').val(project.gerente_proyecto);
            $('#projectSponsor').val(project.sponsor_proyecto);
            $('#projectClient').val(project.client_id);
            $('#projectObjectives').val(project.objetivos);
            $('#projectDeliverables').val(project.entregables);
            $('#projectStatus').val(project.estado);
            $('#projectBudget').val(formatNumberForDisplay(project.budget || 0));
            $('#btnDeleteProject').show();
            $('#activity-log-tab').show();

            // Load activity log
            $.get(`${activityLogApiUrl}?project_id=${project.id}`, function(logs) {
                const logList = $('#activityLogList');
                logList.empty();
                if (logs && logs.length > 0) {
                    logs.forEach(log => {
                        const logItem = `
                            <div class="list-group-item list-group-item-action flex-column align-items-start">
                                <div class="d-flex w-100 justify-content-between">
                                    <h6 class="mb-1 small">${log.user_name}</h6>
                                    <small class="text-muted">${formatDateTime(log.timestamp)}</small>
                                </div>
                                <p class="mb-1 small">${log.action}</p>
                            </div>
                        `;
                        logList.append(logItem);
                    });
                } else {
                    logList.html('<p class="text-muted text-center">No hay actividades registradas para este proyecto.</p>');
                }
            }).fail(function() {
                $('#activityLogList').html('<p class="text-danger text-center">Error al cargar el log de actividad.</p>');
            });

        } else { // Create mode
            $('#projectModalLabel').text('Nuevo Proyecto');
            $('#projectId').val('');
            $('#btnDeleteProject').hide();
            $('#activity-log-tab').hide();
        }
        projectModal.show();
    }

    init();
});