$(document).ready(function() {
    const conversationsList = $('#groups-list');
    const usersList = $('#users-list');
    const newGroupModal = new bootstrap.Modal($('#newGroupModal'));
    const newGroupForm = $('#newGroupForm');
    const groupNameInput = $('#newGroupModal #groupName');
    const userSelect = $('#newGroupModal #userSelect');
    const currentUserEmail = $('#userEmail').val();
    let usersMap = {};
    let allUsers = []; // To store all users for participant selection
    let allConversations = []; // NEW: Global to store all conversations

    // Edit Group Modal elements
    const editGroupModal = new bootstrap.Modal($('#editGroupModal'));
    const editGroupForm = $('#editGroupForm');
    const editGroupIdInput = $('#editGroupId');
    const editGroupNameInput = $('#editGroupName');
    const editGroupPhotoUrlInput = $('#editGroupPhotoUrl');
    const editUserSelect = $('#editUserSelect');
    const btnDeleteGroup = $('#btnDeleteGroup');

    function initialLoad() {
        $.get('api/usuarios.php', function(users) {
            allUsers = users;
            usersMap = users.reduce((map, user) => {
                map[user.email] = user.nombreCompleto;
                return map;
            }, {});

            loadUsersForSidebar(users);
            loadConversations(usersMap);
            setupNewGroupForm(users);
            setupEditGroupFormUsers(users);
        });
    }

    function loadUsersForSidebar(users) {
        usersList.empty();
        users.forEach(function(user) {
            if (user.email !== currentUserEmail) {
                const userItem = $(`<div class="sidebar-item" data-email="${user.email}">${user.nombreCompleto}</div>`);
                userItem.click(function() {
                    openPrivateConversation(user.email, user.nombreCompleto);
                });
                usersList.append(userItem);
            }
        });
    }

    function loadConversations(usersMap) {
        $.get('api/conversations.php', function(conversations) {
            allConversations = conversations;
            conversationsList.empty();
            conversations.forEach(function(conv) {
                let convName;
                let convPhoto = 'https://via.placeholder.com/50';

                if (conv.type === 'group') {
                    convName = conv.name;
                    convPhoto = conv.foto || 'https://via.placeholder.com/50';
                } else {
                    const otherUserEmail = conv.participants.find(p => p !== currentUserEmail);
                    if (otherUserEmail && usersMap[otherUserEmail]) {
                        convName = usersMap[otherUserEmail];
                        const otherUser = allUsers.find(u => u.email === otherUserEmail);
                        convPhoto = otherUser ? (otherUser.foto || 'https://via.placeholder.com/50') : 'https://via.placeholder.com/50';
                    } else {
                        convName = otherUserEmail || 'Usuario no disponible';
                    }
                }

                const conversationItem = $(`
                    <div class="sidebar-item d-flex align-items-center" data-id="${conv.id}">
                        <img src="${convPhoto}" class="rounded-circle me-2" style="width: 30px; height: 30px; object-fit: cover;">
                        <span class="flex-grow-1">${convName}</span>
                        ${conv.type === 'group' ? `<button class="btn btn-sm btn-outline-secondary btn-edit-group" data-id="${conv.id}"><i class="bi bi-pencil"></i></button>` : ''}
                    </div>
                `);
                conversationItem.click(function(e) {
                    if (!$(e.target).hasClass('btn-edit-group') && !$(e.target).closest('.btn-edit-group').length) {
                        window.selectConversation(conv.id, convName);
                        $('.sidebar-item').removeClass('active');
                        $(this).addClass('active');
                    }
                });
                conversationsList.append(conversationItem);
            });
        });
    }
    
    function setupNewGroupForm(users) {
        userSelect.empty();
        users.forEach(function(user) {
            if (user.email !== currentUserEmail) {
                userSelect.append(`<option value="${user.email}">${user.nombreCompleto}</option>`);
            }
        });
    }

    function setupEditGroupFormUsers(users) {
        editUserSelect.empty();
        users.forEach(function(user) {
            editUserSelect.append(`<option value="${user.email}">${user.nombreCompleto}</option>`);
        });
    }

    function openPrivateConversation(userEmail, userName) {
        $('.sidebar-item').removeClass('active');
        $(`.sidebar-item[data-email="${userEmail}"]`).addClass('active');

        $.get(`api/conversations.php?with_user=${userEmail}`)
            .done(function(conversation) {
                window.selectConversation(conversation.id, userName);
            })
            .fail(function() {
                $.ajax({
                    url: 'api/conversations.php',
                    type: 'POST',
                    contentType: 'application/json',
                    data: JSON.stringify({ type: 'private', participants: [userEmail] }),
                    success: function(newConv) {
                        window.selectConversation(newConv.id, userName);
                        loadConversations(usersMap);
                    }
                });
            });
    }

    newGroupForm.submit(function(e) {
        e.preventDefault();
        const name = groupNameInput.val();
        const participants = userSelect.val();
        if (!name || !participants || participants.length === 0) {
            alert('El nombre del grupo y al menos un participante son obligatorios.');
            return;
        }
        $.ajax({
            url: 'api/conversations.php',
            type: 'POST',
            contentType: 'application/json',
            data: JSON.stringify({ type: 'group', name: name, participants: participants }),
            success: function(newConv) {
                loadConversations(usersMap);
                newGroupModal.hide();
                window.selectConversation(newConv.id, newConv.name);
            },
            error: function() {
                alert('Error al crear el grupo.');
            }
        });
    });

    // Open Edit Group Modal
    conversationsList.on('click', '.btn-edit-group', function(e) {
        e.stopPropagation();
        const convId = $(this).data('id');
        const conversationToEdit = allConversations.find(c => c.id === convId);

        if (conversationToEdit && conversationToEdit.type === 'group') {
            editGroupIdInput.val(conversationToEdit.id);
            editGroupNameInput.val(conversationToEdit.name);
            editGroupPhotoUrlInput.val(conversationToEdit.foto || '');
            editUserSelect.val(conversationToEdit.participants); // Select current participants
            editGroupModal.show();
        } else {
            alert('No se pudo encontrar el grupo para editar.');
        }
    });

    // Handle Edit Group Form Submission
    editGroupForm.submit(function(e) {
        e.preventDefault();
        const convId = editGroupIdInput.val();
        const name = editGroupNameInput.val();
        const photo = editGroupPhotoUrlInput.val();
        const participants = editUserSelect.val();

        if (!name || !participants || participants.length === 0) {
            alert('El nombre del grupo y al menos un participante son obligatorios.');
            return;
        }

        $.ajax({
            url: `api/conversations.php?id=${convId}`,
            type: 'PUT',
            contentType: 'application/json',
            data: JSON.stringify({ name: name, participants: participants, foto: photo }),
            success: function(updatedConv) {
                editGroupModal.hide();
                loadConversations(usersMap);
                if (window.currentConversationId === updatedConv.id) {
                    $('#chat-title').text(updatedConv.name);
                }
            },
            error: function(xhr) {
                let errorMessage = 'Error al guardar los cambios del grupo.';
                if (xhr.responseJSON && xhr.responseJSON.error) {
                    errorMessage = xhr.responseJSON.error;
                }
                alert(errorMessage);
            }
        });
    });

    // Handle Delete Group Button
    btnDeleteGroup.on('click', function() {
        const convId = editGroupIdInput.val();
        if (confirm('¿Estás seguro de que quieres eliminar este grupo? Esta acción es irreversible.')) {
            $.ajax({
                url: `api/conversations.php?id=${convId}`,
                type: 'DELETE',
                success: function() {
                    editGroupModal.hide();
                    loadConversations(usersMap);
                },
                error: function(xhr) {
                    let errorMessage = 'Error al eliminar el grupo.';
                    if (xhr.responseJSON && xhr.responseJSON.error) {
                        errorMessage = xhr.responseJSON.error;
                    }
                    alert(errorMessage);
                }
            });
        }
    });

    initialLoad();
});