$(document).ready(function() {
    const tasksApiUrl = 'api/tasks.php';
    const usersApiUrl = 'api/usuarios.php';
    const projectsApiUrl = 'api/projects.php';
    const clientsApiUrl = 'api/clients.php';

    const taskModal = new bootstrap.Modal(document.getElementById('taskModal'));
    const taskForm = $('#taskForm');
    const ganttElement = $('#gantt');
    const ganttTitle = $('#gantt-title');
    const projectDetailsContainer = $('#project-details');
    const addTaskButton = $('[data-bs-target="#taskModal"]');

    const urlParams = new URLSearchParams(window.location.search);
    const projectId = urlParams.get('project_id');

    let gantt;
    let allTasks = []; // Global para almacenar todas las tareas del proyecto
    let allUsers = []; // Global para almacenar todos los usuarios
    let currentTaskWorkLogs = []; // Variable para almacenar los trabajos realizados de la tarea actual

    function formatDateToDDMMYYYY(dateString) {
        if (!dateString) return 'N/A';
        const [year, month, day] = dateString.split('-');
        return `${day}/${month}/${year}`;
    }

    function formatNumber(value) {
        if (typeof value !== 'number') {
            value = parseFloat(value);
        }
        if (isNaN(value)) {
            return 'N/A';
        }
        return value.toLocaleString('es-PY', { minimumFractionDigits: 0, maximumFractionDigits: 2 });
    }

    if (!projectId) {
        ganttElement.html('<div class="alert alert-warning">No se ha especificado un proyecto. Por favor, selecciona un proyecto desde el <a href="index.html" class="alert-link">Dashboard de Proyectos</a>.</div>');
        ganttTitle.text('Sin Proyecto');
        addTaskButton.hide();
        return;
    }

    function init() {
        ganttElement.empty();

        const getTasks = $.get(tasksApiUrl, { project_id: projectId });
        const getUsers = $.get(usersApiUrl);
        const getProjects = $.get(projectsApiUrl);
        const getClients = $.get(clientsApiUrl);

        Promise.all([getTasks, getUsers, getProjects, getClients]).then(function([tasks, users, projects, clients]) {
            allTasks = tasks; // Almacenar todas las tareas
            allUsers = users; // Almacenar todos los usuarios

            const currentProject = projects.find(p => p.id === projectId);
            const currentClient = currentProject ? clients.find(c => c.id === currentProject.client_id) : null;

            renderProjectHeader(currentProject, currentClient);
            populateUserDropdown(users); // Para el modal de tarea
            initializeGantt(tasks);
            renderTaskList(tasks); // Renderizar el listado de tareas
            setupEventListeners();

        }).catch(function() {
            ganttElement.html('<div class="alert alert-danger">Error al cargar los datos. Asegúrese de que los servicios PHP están funcionando correctamente.</div>');
            $('#taskListBody').html('<tr><td colspan="7" class="text-center text-danger">Error al cargar las tareas.</td></tr>');
        });
    }

    function renderProjectHeader(project, client) {
        if (project) {
            ganttTitle.text(`Proyecto: ${project.nombre}`);
            let clientInfo = '<div class="col-md-12"><p class="text-muted">Este proyecto no tiene un cliente asignado.</p></div>';
            if (client) {
                clientInfo = `
                    <div class="col-md-4">
                        <strong class="d-block">Cliente:</strong>
                        <span>${client.nombre_razon_social}</span>
                    </div>
                    <div class="col-md-4">
                        <strong class="d-block">Contacto:</strong>
                        <span>${client.nombre_contacto || 'N/A'}</span>
                    </div>
                    <div class="col-md-4">
                        <strong class="d-block">Email / Teléfono:</strong>
                        <span>${client.correo_contacto || 'N/A'} / ${client.telefono_contacto || 'N/A'}</span>
                    </div>
                `;
            }
            projectDetailsContainer.html(clientInfo);
        } else {
            ganttTitle.text('Proyecto no encontrado');
            projectDetailsContainer.empty();
        }
    }

    function populateUserDropdown(users) {
        const responsibleSelect = $('#taskResponsible');
        responsibleSelect.empty().append('<option value="">Sin asignar</option>');
        users.forEach(user => {
            responsibleSelect.append(`<option value="${user.nombreCompleto}">${user.nombreCompleto}</option>`);
        });
    }

    function getDaysRemainingBadge(days) {
        let badgeClass = 'bg-secondary';
        let badgeText = '';

        if (days === null) {
            badgeText = 'Sin fecha de fin';
        } else if (days < 0) {
            badgeClass = 'bg-danger';
            badgeText = `Vencido hace ${Math.abs(days)} días`;
        } else if (days <= 2) {
            badgeClass = 'bg-warning text-dark';
            badgeText = `Vence en ${days} día${days === 1 ? '' : 's'}`;
        } else {
            badgeClass = 'bg-success';
            badgeText = `Vence en ${days} días`;
        }
        return `<span class="badge ${badgeClass}">${badgeText}</span>`;
    }

    function initializeGantt(tasks) {
        if (!tasks || tasks.length === 0) {
            ganttElement.html('<div class="alert alert-info">Este proyecto aún no tiene tareas. ¡Añade una para comenzar!</div>');
            return;
        }

        gantt = new Gantt("#gantt", tasks, {
            header_height: 50,
            column_width: 30,
            step: 24,
            view_modes: ['Quarter Day', 'Half Day', 'Day', 'Week', 'Month'],
            bar_height: 20,
            bar_corner_radius: 3,
            arrow_curve: 5,
            padding: 18,
            view_mode: 'Week',
            date_format: 'YYYY-MM-DD',
            language: 'es',
            custom_popup_html: function(task) {
                const responsible = task.responsible ? `<p><strong>Responsable:</strong> ${task.responsible}</p>` : '';
                const daysRemainingBadge = task.days_remaining !== undefined ? `<p>${getDaysRemainingBadge(task.days_remaining)}</p>` : '';
                return `
                    <div class="details-container" style="padding:10px; min-width: 200px;">
                        <h5>${task.name}</h5>
                        <hr style="margin: 5px 0;">
                        ${responsible}
                        ${daysRemainingBadge}
                        <p>Progreso: ${formatNumber(task.progress)}%</p>
                        <p>Inicio: ${formatDateToDDMMYYYY(task.start)}</p>
                        <p>Fin: ${formatDateToDDMMYYYY(task.end)}</p>
                        <small class="text-muted">Haz clic para editar</small>
                    </div>
                `;
            },
            on_click: function (task) {
                showTaskModal(task);
            }
        });
    }

    function calculateHours(startTime, endTime) {
        if (!startTime || !endTime) return 0;

        const [startH, startM] = startTime.split(':').map(Number);
        const [endH, endM] = endTime.split(':').map(Number);

        const startDate = new Date(0, 0, 0, startH, startM);
        const endDate = new Date(0, 0, 0, endH, endM);

        let diff = endDate.getTime() - startDate.getTime();
        if (diff < 0) {
            diff += 1000 * 60 * 60 * 24;
        }

        return formatNumber(diff / (1000 * 60 * 60));
    }

    function renderWorkLogs(workLogs) {
        const workLogsList = $('#workLogsList');
        workLogsList.empty();

        if (!workLogs || workLogs.length === 0) {
            workLogsList.append('<p class="text-muted text-center mt-3">No hay trabajos registrados para esta tarea.</p>');
            return;
        }

        workLogs.forEach(log => {
            const logHtml = `
                <div class="list-group-item list-group-item-action flex-column align-items-start mb-2">
                    <div class="d-flex w-100 justify-content-between">
                        <h6 class="mb-1">${log.type} - ${formatDateToDDMMYYYY(log.date)} (${formatNumber(log.hours)} horas)</h6>
                        <small>
                            <button type="button" class="btn btn-sm btn-outline-primary edit-work-log" data-id="${log.id}">Editar</button>
                            <button type="button" class="btn btn-sm btn-outline-danger delete-work-log" data-id="${log.id}">Eliminar</button>
                        </small>
                    </div>
                    <p class="mb-1">${log.description}</p>
                    ${log.link ? `<small><a href="${log.link}" target="_blank">Ver archivos</a></small>` : ''}
                </div>
            `;
            workLogsList.append(logHtml);
        });
    }

    function filterTasks() {
        const filterTaskName = $('#filterTaskName').val().toLowerCase();
        const filterTaskStart = $('#filterTaskStart').val();
        const filterTaskEnd = $('#filterTaskEnd').val();
        const filterTaskProgress = $('#filterTaskProgress').val().trim();
        const filterTaskResponsible = $('#filterTaskResponsibleText').val().toLowerCase();
        const filterTaskDependencies = $('#filterTaskDependencies').val().toLowerCase();

        const filteredTasks = allTasks.filter(task => {
            const taskName = String(task.name || '').toLowerCase();
            const taskStart = task.start;
            const taskEnd = task.end;
            const taskProgress = parseInt(task.progress, 10);
            const taskResponsible = String(task.responsible || '').toLowerCase();
            const taskDependencies = String(task.dependencies || '').toLowerCase();

            if (filterTaskName && !taskName.includes(filterTaskName)) return false;
            if (filterTaskResponsible && !taskResponsible.includes(filterTaskResponsible)) return false;
            if (filterTaskDependencies && !taskDependencies.includes(filterTaskDependencies)) return false;

            if (filterTaskStart && taskEnd < filterTaskStart) return false;
            if (filterTaskEnd && taskStart > filterTaskEnd) return false;

            if (filterTaskProgress !== '') {
                let operator = filterTaskProgress.charAt(0);
                let value;

                if (operator === '>' || operator === '<') {
                    value = parseInt(filterTaskProgress.substring(1), 10);
                    if (isNaN(value)) return true;

                    if (operator === '>' && taskProgress <= value) return false;
                    if (operator === '<' && taskProgress >= value) return false;

                } else {
                    value = parseInt(filterTaskProgress, 10);
                    if (isNaN(value)) return true;

                    if (taskProgress !== value) return false;
                }
            }

            return true;
        });

        renderTaskList(filteredTasks);
    }

    function setupEventListeners() {
        addTaskButton.off('click').on('click', function() {
            showTaskModal(null);
        });

        $('#addWorkLogBtn').off('click').on('click', function() {
            const workLogId = $(this).data('edit-id');
            const workLogType = $('#workLogType').val();
            const workLogDate = $('#workLogDate').val();
            const workLogStartTime = $('#workLogStartTime').val();
            const workLogEndTime = $('#workLogEndTime').val();
            const workLogDescription = $('#workLogDescription').val();
            const workLogLink = $('#workLogLink').val();

            if (!workLogType || !workLogDate || !workLogStartTime || !workLogEndTime) {
                alert('Por favor, completa el tipo, fecha, hora de inicio y hora de fin del trabajo.');
                return;
            }

            const hours = calculateHours(workLogStartTime, workLogEndTime);

            const newWorkLog = {
                id: workLogId || 'WorkLog_' + Date.now(),
                type: workLogType,
                date: workLogDate,
                start_time: workLogStartTime,
                end_time: workLogEndTime,
                hours: hours,
                description: workLogDescription,
                link: workLogLink
            };

            if (workLogId) {
                const index = currentTaskWorkLogs.findIndex(log => log.id === workLogId);
                if (index !== -1) {
                    currentTaskWorkLogs[index] = newWorkLog;
                }
                $(this).removeData('edit-id').text('Agregar Trabajo');
            } else {
                currentTaskWorkLogs.push(newWorkLog);
            }
            
            renderWorkLogs(currentTaskWorkLogs);
            $('#workLogType').val('');
            $('#workLogDate').val('');
            $('#workLogStartTime').val('');
            $('#workLogEndTime').val('');
            $('#workLogDescription').val('');
            $('#workLogLink').val('');
        });

        $('#workLogsList').on('click', '.edit-work-log', function() {
            const logId = $(this).data('id');
            const logToEdit = currentTaskWorkLogs.find(log => log.id === logId);

            if (logToEdit) {
                $('#workLogType').val(logToEdit.type);
                $('#workLogDate').val(logToEdit.date);
                $('#workLogStartTime').val(logToEdit.start_time);
                $('#workLogEndTime').val(logToEdit.end_time);
                $('#workLogDescription').val(logToEdit.description);
                $('#workLogLink').val(logToEdit.link);
                $('#addWorkLogBtn').text('Actualizar Trabajo').data('edit-id', logId);
            }
        });

        $('#workLogsList').on('click', '.delete-work-log', function() {
            const logId = $(this).data('id');
            if (confirm('¿Estás seguro de que quieres eliminar este trabajo realizado?')) {
                currentTaskWorkLogs = currentTaskWorkLogs.filter(log => log.id !== logId);
                renderWorkLogs(currentTaskWorkLogs);
            }
        });

        taskForm.off('submit').on('submit', function(e) {
            e.preventDefault();
            const taskId = $('#taskId').val();
            
            const taskData = {
                id: taskId || 'Task_' + Date.now(),
                project_id: projectId,
                name: $('#taskName').val(),
                start: $('#taskStart').val(),
                end: $('#taskEnd').val(),
                progress: parseInt($('#taskProgress').val(), 10) || 0,
                responsible: $('#taskResponsible').val(),
                dependencies: $('#taskDependencies').val(),
                trabajos_realizados: currentTaskWorkLogs
            };

            const isEditing = !!taskId;

            $.ajax({
                url: isEditing ? `${tasksApiUrl}?id=${taskId}` : tasksApiUrl,
                type: isEditing ? 'PUT' : 'POST',
                contentType: 'application/json',
                data: JSON.stringify(taskData),
                success: function() {
                    taskModal.hide();
                    init();
                },
                error: function() { alert('Error al guardar la tarea.'); }
            });
        });

        $('#btnDeleteTask').off('click').on('click', function() {
            const taskId = $('#taskId').val();
            if (taskId && confirm('¿Estás seguro de que quieres eliminar esta tarea?')) {
                $.ajax({
                    url: `${tasksApiUrl}?id=${taskId}`,
                    type: 'DELETE',
                    success: function() {
                        taskModal.hide();
                        init();
                    },
                    error: function() { alert('Error al eliminar la tarea.'); }
                });
            }
        });

        $('.task-filter').on('input change', filterTasks);
        $('#clearFilters').on('click', function() {
            $('#filterTaskName').val('');
            $('#filterTaskStart').val('');
            $('#filterTaskEnd').val('');
            $('#filterTaskProgress').val('');
            $('#filterTaskResponsibleText').val('');
            $('#filterTaskDependencies').val('');
            filterTasks();
        });
    }

    function showTaskModal(task) {
        taskForm[0].reset();
        currentTaskWorkLogs = [];
        $('#workLogsList').empty().append('<p class="text-muted text-center mt-3">No hay trabajos registrados para esta tarea.</p>');
        $('#addWorkLogBtn').text('Agregar Trabajo').removeData('edit-id');

        $('#task-details-tab').tab('show');

        if (task) {
            $('#taskModalLabel').text('Editar Tarea');
            $('#taskId').val(task.id);
            $('#taskName').val(task.name);
            $('#taskStart').val(task.start);
            $('#taskEnd').val(task.end);
            $('#taskProgress').val(task.progress);
            $('#taskResponsible').val(task.responsible);
            $('#taskDependencies').val(task.dependencies);
            $('#btnDeleteTask').show();

            if (task.trabajos_realizados && Array.isArray(task.trabajos_realizados)) {
                currentTaskWorkLogs = [...task.trabajos_realizados];
                renderWorkLogs(currentTaskWorkLogs);
            }
        } else {
            $('#taskModalLabel').text('Añadir Tarea');
            $('#taskId').val('');
            $('#btnDeleteTask').hide();
        }
        taskModal.show();
    }

    function renderTaskList(tasks) {
        const taskListBody = $('#taskListBody');
        taskListBody.empty();

        if (!tasks || tasks.length === 0) {
            taskListBody.append('<tr><td colspan="7" class="text-center text-muted">No se encontraron tareas que coincidan con los filtros.</td></tr>');
            return;
        }

        tasks.forEach(task => {
            const responsibleUser = allUsers.find(user => user.nombreCompleto === task.responsible);
            const responsibleName = responsibleUser ? responsibleUser.nombreCompleto : (task.responsible || 'Sin asignar');

            const row = `
                <tr>
                    <td>${task.name}</td>
                    <td>${formatDateToDDMMYYYY(task.start)}</td>
                    <td>${formatDateToDDMMYYYY(task.end)}</td>
                    <td>${task.progress}%</td>
                    <td>${responsibleName}</td>
                    <td>${task.dependencies || '-'}</td>
                    <td>
                        <button type="button" class="btn btn-sm btn-info edit-task-btn" data-task-id="${task.id}">Editar</button>
                        <button type="button" class="btn btn-sm btn-danger delete-task-btn" data-task-id="${task.id}">Eliminar</button>
                    </td>
                </tr>
            `;
            taskListBody.append(row);
        });

        $('.edit-task-btn').off('click').on('click', function() {
            const taskId = $(this).data('task-id');
            const taskToEdit = allTasks.find(t => t.id === taskId);
            if (taskToEdit) {
                showTaskModal(taskToEdit);
            }
        });

        $('.delete-task-btn').off('click').on('click', function() {
            const taskId = $(this).data('task-id');
            if (confirm('¿Estás seguro de que quieres eliminar esta tarea?')) {
                deleteTask(taskId);
            }
        });
    }

    function deleteTask(taskId) {
        $.ajax({
            url: `${tasksApiUrl}?id=${taskId}`,
            type: 'DELETE',
            success: function() {
                alert('Tarea eliminada con éxito.');
                init();
            },
            error: function() {
                alert('Error al eliminar la tarea.');
            }
        });
    }

    init();
});