<?php
session_start();
header('Content-Type: application/json');

// --- NEW PERMISSION CHECK FUNCTION ---
function checkPermission($requiredRole) {
    if (!isset($_SESSION['user_role'])) {
        http_response_code(401); // Unauthorized
        echo json_encode(['error' => 'No autenticado.']);
        exit();
    }

    $userRole = $_SESSION['user_role'];
    $rolesHierarchy = [
        'admin' => 3,
        'project_manager' => 2,
        'team_member' => 1
    ];

    if (!isset($rolesHierarchy[$userRole]) || !isset($rolesHierarchy[$requiredRole])) {
        http_response_code(500); // Internal Server Error - invalid role configured
        echo json_encode(['error' => 'Error de configuración de roles.']);
        exit();
    }

    if ($rolesHierarchy[$userRole] < $rolesHierarchy[$requiredRole]) {
        http_response_code(403); // Forbidden
        echo json_encode(['error' => 'Permiso denegado. Rol insuficiente.']);
        exit();
    }
}
// --- END NEW PERMISSION CHECK FUNCTION ---

$filePath = '../data/usuarios.json';

// Asegurarse de que el archivo exista y no esté vacío
if (!file_exists($filePath) || filesize($filePath) === 0) {
    file_put_contents($filePath, '[]');
}

$method = $_SERVER['REQUEST_METHOD'];
$id = isset($_GET['id']) ? $_GET['id'] : null;

// Función para leer usuarios
function getUsers($filePath) {
    $fileContent = file_get_contents($filePath);
    $users = json_decode($fileContent, true);
    if (json_last_error() !== JSON_ERROR_NONE) {
        return [];
    }
    return $users;
}

// Función para guardar usuarios
function saveUsers($filePath, $users) {
    $fileHandle = fopen($filePath, 'w');
    if (flock($fileHandle, LOCK_EX)) {
        fwrite($fileHandle, json_encode($users, JSON_PRETTY_PRINT));
        flock($fileHandle, LOCK_UN);
    }
    fclose($fileHandle);
}

switch ($method) {
    case 'GET':
        // GET is generally allowed for all to view user list
        echo json_encode(getUsers($filePath));
        break;

    case 'PUT':
    case 'POST': // Handle POST for PUT override
        if ($method === 'POST') {
            $input = json_decode(file_get_contents('php://input'), true);
            if (json_last_error() !== JSON_ERROR_NONE) {
                http_response_code(400);
                echo json_encode(['error' => 'Invalid JSON']);
                exit;
            }
        } else { // This is a real PUT request (from admin user management)
            $input = json_decode(file_get_contents('php://input'), true);
            if (json_last_error() !== JSON_ERROR_NONE) {
                http_response_code(400);
                echo json_encode(['error' => 'Invalid JSON']);
                exit;
            }
        }

        if ($method === 'POST') {
            // This is a real POST request for creating a user
            checkPermission('admin'); // --- PERMISSION CHECK for creating users ---

            $users = getUsers($filePath);
            $input['id'] = uniqid('user_', true);

            // Hash password if provided for new user
            if (isset($input['password']) && !empty($input['password'])) {
                $input['password'] = password_hash($input['password'], PASSWORD_DEFAULT);
            } else {
                // If no password provided for new user, remove the key so it's not stored as empty/null
                unset($input['password']);
            }
            
            // Ensure new user has a role, default to team_member if not provided
            $input['role'] = $input['role'] ?? 'team_member';

            // Ensure email notification preference is set, default to true
            $input['email_notifications_enabled'] = $input['email_notifications_enabled'] ?? true;

            $users[] = $input;
            saveUsers($filePath, $users);

            http_response_code(201);
            echo json_encode($input);
            break;
        }

        // --- Logic for PUT method ---
        if (!$id) {
            http_response_code(400);
            echo json_encode(['error' => 'User ID is required']);
            exit;
        }

        $users = getUsers($filePath);
        $userFound = false;
        $userToEdit = null;
        $updatedUser = null;

        foreach ($users as $user) {
            if ($user['id'] === $id) {
                $userToEdit = $user;
                break;
            }
        }

        if (!$userToEdit) {
            http_response_code(404);
            echo json_encode(['error' => 'User not found']);
            exit;
        }

        // --- PERMISSION CHECK FOR PUT ---
        // Allow if the user is an admin OR if the user is editing their own profile.
        if ($_SESSION['user_role'] !== 'admin' && $userToEdit['email'] !== $_SESSION['user_email']) {
            http_response_code(403);
            echo json_encode(['error' => 'Permiso denegado. No puede editar el perfil de otro usuario.']);
            exit;
        }

        foreach ($users as $key => $user) {
            if ($user['id'] === $id) {
                // Handle password update: only if new password is provided
                if (isset($input['password']) && !empty($input['password'])) {
                    $input['password'] = password_hash($input['password'], PASSWORD_DEFAULT);
                } else {
                    // If password field is empty, remove it from input to prevent overwriting existing hash
                    unset($input['password']);
                }
                
                // If a non-admin is editing, prevent them from changing their role or hourly rate
                if ($_SESSION['user_role'] !== 'admin') {
                    unset($input['role']);
                    unset($input['hourly_rate']);
                } else {
                    // Admin can change role, ensure it's preserved if not provided
                    $input['role'] = $input['role'] ?? $user['role'] ?? 'team_member';
                }

                // Ensure email notification preference is preserved or updated
                $input['email_notifications_enabled'] = $input['email_notifications_enabled'] ?? $user['email_notifications_enabled'] ?? true;

                $users[$key] = array_merge($user, $input);
                $users[$key]['id'] = $id;
                $updatedUser = $users[$key];
                $userFound = true;
                break;
            }
        }

        if ($userFound) {
            saveUsers($filePath, $users);
            echo json_encode($updatedUser);
        } else {
            http_response_code(404);
            echo json_encode(['error' => 'User not found']);
        }
        break;

    case 'DELETE':
        checkPermission('admin'); // --- PERMISSION CHECK ---
        if (!$id) {
            http_response_code(400);
            echo json_encode(['error' => 'User ID is required']);
            exit;
        }

        $users = getUsers($filePath);
        $initialCount = count($users);
        $users = array_filter($users, function($user) use ($id) {
            return $user['id'] !== $id;
        });

        if (count($users) < $initialCount) {
            saveUsers($filePath, array_values($users));
            http_response_code(204); // No Content
        } else {
            http_response_code(404);
            echo json_encode(['error' => 'User not found']);
        }
        break;

    default:
        http_response_code(405);
        echo json_encode(['error' => 'Method Not Allowed']);
        break;
}
?>