<?php
session_start();
header('Content-Type: application/json');

// --- PERMISSION CHECK FUNCTION (Copied from usuarios.php) ---
function checkPermission($requiredRole) {
    if (!isset($_SESSION['user_role'])) {
        http_response_code(401); // Unauthorized
        echo json_encode(['error' => 'No autenticado.']);
        exit();
    }

    $userRole = $_SESSION['user_role'];
    $rolesHierarchy = [
        'admin' => 3,
        'project_manager' => 2,
        'team_member' => 1
    ];

    if (!isset($rolesHierarchy[$userRole]) || !isset($rolesHierarchy[$requiredRole])) {
        http_response_code(500); // Internal Server Error - invalid role configured
        echo json_encode(['error' => 'Error de configuración de roles.']);
        exit();
    }

    if ($rolesHierarchy[$userRole] < $rolesHierarchy[$requiredRole]) {
        http_response_code(403); // Forbidden
        echo json_encode(['error' => 'Permiso denegado. Rol insuficiente.']);
        exit();
    }
}
// --- END PERMISSION CHECK FUNCTION ---

// Include notification helper function
require_once 'notification_helper.php';
require_once 'activity_log_helper.php';

$filePath = '../data/tasks.json';

if (!file_exists($filePath) || filesize($filePath) === 0) {
    file_put_contents($filePath, '[]');
}

$method = $_SERVER['REQUEST_METHOD'];
$id = isset($_GET['id']) ? $_GET['id'] : null;
$projectId = isset($_GET['project_id']) ? $_GET['project_id'] : null;

function getTasks($filePath) {
    $fileContent = file_get_contents($filePath);
    $tasks = json_decode($fileContent, true);
    return json_last_error() === JSON_ERROR_NONE ? $tasks : [];
}

function saveTasks($filePath, $tasks) {
    $fileHandle = fopen($filePath, 'w');
    if (flock($fileHandle, LOCK_EX)) {
        fwrite($fileHandle, json_encode($tasks, JSON_PRETTY_PRINT));
        flock($fileHandle, LOCK_UN);
    }
    fclose($fileHandle);
}

switch ($method) {
    case 'GET':
        $tasks = getTasks($filePath);
        if ($projectId) {
            $filteredTasks = array_filter($tasks, function($task) use ($projectId) {
                return isset($task['project_id']) && $task['project_id'] === $projectId;
            });
            $tasks = array_values($filteredTasks); // Use filtered tasks for further processing
        }

        $today = new DateTime(); // Get current date

        foreach ($tasks as $key => $task) {
            // Calculate days_remaining (NEW LOGIC)
            if (isset($task['end']) && !empty($task['end'])) {
                try {
                    $endDate = new DateTime($task['end']);
                    $interval = $today->diff($endDate);
                    $daysRemaining = (int)$interval->format('%R%a'); // %R for sign (+/-), %a for total days
                    $tasks[$key]['days_remaining'] = $daysRemaining;
                } catch (Exception $e) {
                    $tasks[$key]['days_remaining'] = null; // Handle invalid date format
                }
            } else {
                $tasks[$key]['days_remaining'] = null; // No end date
            }

            // Format hours in trabajos_realizados
            if (isset($tasks[$key]['trabajos_realizados']) && is_array($tasks[$key]['trabajos_realizados'])) {
                foreach ($tasks[$key]['trabajos_realizados'] as $workLogKey => $workLog) {
                    if (isset($workLog['hours'])) {
                        $tasks[$key]['trabajos_realizados'][$workLogKey]['hours'] = (float)$workLog['hours']; // Ensure float
                    }
                }
            }
        }
        $userEmail = isset($_GET['user_email']) ? $_GET['user_email'] : null;

        if ($userEmail) {
            $tasks = array_filter($tasks, function($task) use ($userEmail) {
                return isset($task['responsible_email']) && $task['responsible_email'] === $userEmail;
            });
            $tasks = array_values($tasks); // Re-index the array
        }

        echo json_encode($tasks);
        break;

    case 'POST':
        checkPermission('project_manager'); // Only PMs and Admins can create tasks
        $input = json_decode(file_get_contents('php://input'), true);
        if (!$input) {
            http_response_code(400);
            echo json_encode(['error' => 'Invalid JSON']);
            exit;
        }

        $tasks = getTasks($filePath);
        if (!isset($input['id']) || empty($input['id'])) {
            $input['id'] = uniqid('task_');
        }
        $tasks[] = $input;
        saveTasks($filePath, $tasks);

        // --- Log Activity ---
        logActivity($input['project_id'], "creó la tarea '{$input['name']}'.");

        // --- Create Notification on Assignment ---
        $responsibleEmail = $input['responsible_email'] ?? null;
        // Notify only if the responsible user is not the one creating the task
        if ($responsibleEmail && $responsibleEmail !== $_SESSION['user_email']) {
            $notificationMessage = "Se te ha asignado la nueva tarea: '{$input['name']}'.";
            $notificationLink = "tareas.php?project_id={$input['project_id']}";
            createNotification('../data/notifications.json', $responsibleEmail, $notificationMessage, $notificationLink);
        }

        http_response_code(201);
        echo json_encode($input);
        break;

    case 'PUT':
        // Permission check for PUT is now handled inside the logic
        if (!$id) {
            http_response_code(400);
            echo json_encode(['error' => 'Task ID is required']);
            exit;
        }

        $input = json_decode(file_get_contents('php://input'), true);
        if (!$input) {
            http_response_code(400);
            echo json_encode(['error' => 'Invalid JSON']);
            exit;
        }

        $tasks = getTasks($filePath);
        $taskFound = false;
        $originalTask = null; // Store original task to check responsible

        foreach ($tasks as $key => $task) {
            if ($task['id'] === $id) {
                $originalTask = $task;
                break;
            }
        }

        if (!$originalTask) {
            http_response_code(404);
            echo json_encode(['error' => 'Task not found']);
            exit;
        }

        // --- PERMISSION CHECK FOR PUT (More complex) ---
        $userRole = $_SESSION['user_role'] ?? 'guest';
        $userEmail = $_SESSION['user_email'] ?? '';

        $canEdit = false;
        if ($userRole === 'admin' || $userRole === 'project_manager') {
            $canEdit = true;
        } elseif ($userRole === 'team_member') {
            // Team members can edit their own tasks, but with restrictions
            if (isset($originalTask['responsible']) && $originalTask['responsible'] === $_SESSION['user_name']) {
                $canEdit = true;
                // Prevent team members from changing critical fields
                unset($input['project_id']);
                unset($input['parent_id']);
                unset($input['responsible']);
                unset($input['dependencies']);
            }
        }

        if (!$canEdit) {
            http_response_code(403); // Forbidden
            echo json_encode(['error' => 'Permiso denegado. No tiene permisos para editar esta tarea.']);
            exit;
        }
        // --- END PERMISSION CHECK FOR PUT ---

        foreach ($tasks as $key => $task) {
            if ($task['id'] === $id) {
                $tasks[$key] = array_merge($task, $input);
                $tasks[$key]['id'] = $id; // Asegurar que el ID no cambie
                $updatedTask = $tasks[$key];
                $taskFound = true;
                break;
            }
        }

        if ($taskFound) {
            saveTasks($filePath, $tasks);

            // --- Log Activity ---
            logActivity($updatedTask['project_id'], "actualizó la tarea '{$updatedTask['name']}'.");

            // --- Create Notification on Re-assignment ---
            $newResponsibleEmail = $input['responsible_email'] ?? null;
            $originalResponsibleEmail = $originalTask['responsible_email'] ?? null;

            // Notify if the responsible person has changed and is not the person making the change
            if ($newResponsibleEmail && $newResponsibleEmail !== $originalResponsibleEmail && $newResponsibleEmail !== $_SESSION['user_email']) {
                $notificationMessage = "Se te ha asignado la tarea: '{$updatedTask['name']}'.";
                $notificationLink = "tareas.php?project_id={$updatedTask['project_id']}";
                createNotification('../data/notifications.json', $newResponsibleEmail, $notificationMessage, $notificationLink);
            } else if ($newResponsibleEmail && $newResponsibleEmail === $originalResponsibleEmail && $newResponsibleEmail !== $_SESSION['user_email']) {
                // Optional: Notify on general update if you want to keep this behavior
            }

            echo json_encode($updatedTask);
        } else {
            http_response_code(404);
            echo json_encode(['error' => 'Task not found']);
        }
        break;

    case 'DELETE':
        checkPermission('project_manager'); // Only PMs and Admins can delete tasks
        if (!$id) {
            http_response_code(400);
            echo json_encode(['error' => 'Task ID is required']);
            exit;
        }

        $tasks = getTasks($filePath);
        $taskToDelete = null;
        foreach ($tasks as $task) {
            if ($task['id'] === $id) {
                $taskToDelete = $task;
                break;
            }
        }

        $initialCount = count($tasks);
        $tasks = array_filter($tasks, function($task) use ($id) {
            return $task['id'] !== $id;
        });

        if (count($tasks) < $initialCount && $taskToDelete) {
            logActivity($taskToDelete['project_id'], "eliminó la tarea '{$taskToDelete['name']}'.");
            saveTasks($filePath, array_values($tasks));
            http_response_code(204); // No Content
        } else {
            http_response_code(404);
            echo json_encode(['error' => 'Task not found']);
        }
        break;

    default:
        http_response_code(405);
        echo json_encode(['error' => 'Method Not Allowed']);
        break;
}
?>