<?php
session_start();
header('Content-Type: application/json');

// --- PERMISSION CHECK FUNCTION (Copied from usuarios.php) ---
function checkPermission($requiredRole) {
    if (!isset($_SESSION['user_role'])) {
        http_response_code(401); // Unauthorized
        echo json_encode(['error' => 'No autenticado.']);
        exit();
    }

    $userRole = $_SESSION['user_role'];
    $rolesHierarchy = [
        'admin' => 3,
        'project_manager' => 2,
        'team_member' => 1
    ];

    if (!isset($rolesHierarchy[$userRole]) || !isset($rolesHierarchy[$requiredRole])) {
        http_response_code(500); // Internal Server Error - invalid role configured
        echo json_encode(['error' => 'Error de configuración de roles.']);
        exit();
    }

    if ($rolesHierarchy[$userRole] < $rolesHierarchy[$requiredRole]) {
        http_response_code(403); // Forbidden
        echo json_encode(['error' => 'Permiso denegado. Rol insuficiente.']);
        exit();
    }
}
// --- END PERMISSION CHECK FUNCTION ---

$tasksFilePath = '../data/tasks.json';
$usersFilePath = '../data/usuarios.json';

function getJsonData($filePath) {
    if (!file_exists($filePath) || filesize($filePath) === 0) {
        return [];
    }
    $fileContent = file_get_contents($filePath);
    $data = json_decode($fileContent, true);
    return json_last_error() === JSON_ERROR_NONE ? $data : [];
}

$method = $_SERVER['REQUEST_METHOD'];

switch ($method) {
    case 'GET':
        checkPermission('team_member'); // --- PERMISSION CHECK ---
        $allTasks = getJsonData($tasksFilePath);
        $allUsers = getJsonData($usersFilePath);

        $reportData = [];

        foreach ($allUsers as $user) {
            $userFullName = $user['nombreCompleto'] ?? 'Desconocido';
            $userEmail = $user['email'] ?? '';

            $assignedTasks = array_filter($allTasks, function($task) use ($userFullName) {
                return isset($task['responsible']) && $task['responsible'] === $userFullName;
            });

            $completedTasks = array_filter($assignedTasks, function($task) {
                return isset($task['progress']) && (int)$task['progress'] === 100;
            });

            $pendingTasks = array_filter($assignedTasks, function($task) {
                return isset($task['progress']) && (int)$task['progress'] < 100;
            });

            $totalLoggedHours = 0;
            foreach ($assignedTasks as $task) {
                if (isset($task['trabajos_realizados']) && is_array($task['trabajos_realizados'])) {
                    foreach ($task['trabajos_realizados'] as $workLog) {
                        $totalLoggedHours += (float)($workLog['hours'] ?? 0);
                    }
                }
            }

            $reportData[] = [
                'user_name' => $userFullName,
                'user_email' => $userEmail,
                'assigned_tasks_count' => count($assignedTasks),
                'completed_tasks_count' => count($completedTasks),
                'pending_tasks_count' => count($pendingTasks),
                'total_logged_hours' => round($totalLoggedHours, 2) // Round to 2 decimal places
            ];
        }

        echo json_encode($reportData);
        break;

    default:
        http_response_code(405);
        echo json_encode(['error' => 'Method Not Allowed']);
        break;
}
?>