<?php
session_start();
header('Content-Type: application/json');

// --- PERMISSION CHECK FUNCTION (Copied from usuarios.php) ---
function checkPermission($requiredRole) {
    if (!isset($_SESSION['user_role'])) {
        http_response_code(401); // Unauthorized
        echo json_encode(['error' => 'No autenticado.']);
        exit();
    }

    $userRole = $_SESSION['user_role'];
    $rolesHierarchy = [
        'admin' => 3,
        'project_manager' => 2,
        'team_member' => 1
    ];

    if (!isset($rolesHierarchy[$userRole]) || !isset($rolesHierarchy[$requiredRole])) {
        http_response_code(500); // Internal Server Error - invalid role configured
        echo json_encode(['error' => 'Error de configuración de roles.']);
        exit();
    }

    if ($rolesHierarchy[$userRole] < $rolesHierarchy[$requiredRole]) {
        http_response_code(403); // Forbidden
        echo json_encode(['error' => 'Permiso denegado. Rol insuficiente.']);
        exit();
    }
}
// --- END PERMISSION CHECK FUNCTION ---

// Include notification helper function
require_once 'notification_helper.php';
require_once 'activity_log_helper.php';

$filePath = '../data/projects.json';

if (!file_exists($filePath) || filesize($filePath) === 0) {
    file_put_contents($filePath, '[]');
}

$method = $_SERVER['REQUEST_METHOD'];
$id = isset($_GET['id']) ? $_GET['id'] : null;

function getProjects($filePath) {
    $fileContent = file_get_contents($filePath);
    $projects = json_decode($fileContent, true);
    return json_last_error() === JSON_ERROR_NONE ? $projects : [];
}

function saveProjects($filePath, $projects) {
    $fileHandle = fopen($filePath, 'w');
    if (flock($fileHandle, LOCK_EX)) {
        fwrite($fileHandle, json_encode($projects, JSON_PRETTY_PRINT));
        flock($fileHandle, LOCK_UN);
    }
    fclose($fileHandle);
}

switch ($method) {
    case 'GET':
        $projects = getProjects($filePath);
        $tasks = json_decode(file_get_contents('../data/tasks.json'), true); // Load all tasks
        $allExpenses = json_decode(file_get_contents('../data/expenses.json'), true); // Load all expenses
        $users = json_decode(file_get_contents('../data/usuarios.json'), true); // Load all users

        $userHourlyRates = [];
        foreach ($users as $user) {
            $userHourlyRates[$user['email']] = $user['hourly_rate'] ?? 0;
        }

        $today = new DateTime(); // Get current date

        foreach ($projects as $key => $project) {
            // Calculate progress_general (existing logic)
            $projectTasks = array_filter($tasks, function($task) use ($project) {
                return isset($task['project_id']) && $task['project_id'] === $project['id'];
            });

            // Add tasks to the project object
            $projects[$key]['tasks'] = array_values($projectTasks);

            // Integrate work logs into tasks
            foreach ($projects[$key]['tasks'] as $taskKey => $task) {
                $taskWorkLogs = $task['trabajos_realizados'] ?? [];
                $taskTotalWorkLogCost = 0;
                foreach ($taskWorkLogs as $logKey => $log) {
                    if (isset($log['date'])) {
                        try {
                            $dateObj = new DateTime($log['date']);
                            $taskWorkLogs[$logKey]['date'] = $dateObj->format('d/m/Y');
                        } catch (Exception $e) {}
                    }
                    $responsibleEmail = $projects[$key]['tasks'][$taskKey]['responsible_email'] ?? null;
                    if ($responsibleEmail && isset($log['hours'])) {
                        $hourlyRate = $userHourlyRates[$responsibleEmail] ?? 0;
                        $workLogCost = (float)$log['hours'] * $hourlyRate;
                        $taskWorkLogs[$logKey]['cost'] = $workLogCost;
                        $taskTotalWorkLogCost += $workLogCost;
                    } else {
                        $taskWorkLogs[$logKey]['cost'] = 0;
                    }
                    $taskWorkLogs[$logKey]['hours'] = (float)$log['hours'];
                }
                $projects[$key]['tasks'][$taskKey]['trabajos_realizados'] = array_values($taskWorkLogs);
                $projects[$key]['tasks'][$taskKey]['total_work_log_cost'] = $taskTotalWorkLogCost;

                if (isset($projects[$key]['tasks'][$taskKey]['start'])) {
                    try {
                        $startDateObj = new DateTime($projects[$key]['tasks'][$taskKey]['start']);
                        $projects[$key]['tasks'][$taskKey]['start'] = $startDateObj->format('d/m/Y');
                    } catch (Exception $e) {}
                }
                if (isset($projects[$key]['tasks'][$taskKey]['end'])) {
                    try {
                        $endDateObj = new DateTime($projects[$key]['tasks'][$taskKey]['end']);
                        $projects[$key]['tasks'][$taskKey]['end'] = $endDateObj->format('d/m/Y');
                    } catch (Exception $e) {}
                }
            }

            $totalProgress = 0;
            $taskCount = count($projectTasks);

            if ($taskCount > 0) {
                foreach ($projectTasks as $task) {
                    $totalProgress += $task['progress'] ?? 0;
                }
                $projects[$key]['progress_general'] = round($totalProgress / $taskCount);
            } else {
                $projects[$key]['progress_general'] = 0;
            }

            if (isset($project['terminos']) && !empty($project['terminos'])) {
                try {
                    $endDate = new DateTime($project['terminos']);
                    $interval = $today->diff($endDate);
                    $daysRemaining = (int)$interval->format('%R%a');
                    $projects[$key]['days_remaining'] = $daysRemaining;
                } catch (Exception $e) {
                    $projects[$key]['days_remaining'] = null;
                }
            } else {
                $projects[$key]['days_remaining'] = null;
            }

            $projectBudget = $project['budget'] ?? 0;
            $projectExpenses = array_filter($allExpenses, function($expense) use ($project) {
                return isset($expense['project_id']) && $expense['project_id'] === $project['id'];
            });

            $totalProjectExpenses = 0;
            foreach ($projectExpenses as $expense) {
                $totalProjectExpenses += $expense['amount'] ?? 0;
            }

            $totalProjectWorkLogCost = 0;
            foreach ($projects[$key]['tasks'] as $task) {
                $totalProjectWorkLogCost += $task['total_work_log_cost'] ?? 0;
            }

            $totalProjectCost = $totalProjectExpenses + $totalProjectWorkLogCost;
            $projectBalance = $projectBudget - $totalProjectCost;

            $projects[$key]['financials'] = [
                'budget' => $projectBudget,
                'total_expenses' => $totalProjectExpenses,
                'total_work_log_cost' => $totalProjectWorkLogCost,
                'total_cost' => $totalProjectCost,
                'balance' => $projectBalance,
            ];
        }
        $userEmail = isset($_GET['user_email']) ? $_GET['user_email'] : null;

        if ($userEmail) {
            $projects = array_filter($projects, function($project) use ($userEmail) {
                return (isset($project['gerente_proyecto_email']) && $project['gerente_proyecto_email'] === $userEmail) ||
                       (isset($project['sponsor_proyecto_email']) && $project['sponsor_proyecto_email'] === $userEmail);
            });
            $projects = array_values($projects);
        }

        echo json_encode($projects);
        break;

    case 'POST':
        checkPermission('project_manager'); // --- PERMISSION CHECK ---
        $input = json_decode(file_get_contents('php://input'), true);
        if (!$input) {
            http_response_code(400);
            echo json_encode(['error' => 'Invalid JSON']);
            exit;
        }

        $projects = getProjects($filePath);
        $input['id'] = 'proj_' . time();
        $input['budget'] = $input['budget'] ?? 0;
        $projects[] = $input;
        saveProjects($filePath, $projects);

        // --- Log Activity ---
        logActivity($input['id'], "creó el proyecto '{$input['nombre']}'.");

        // --- Create Notifications for Admins and Project Managers ---
        $users = json_decode(file_get_contents('../data/usuarios.json'), true) ?: [];
        $creatorEmail = $_SESSION['user_email'];
        $notificationMessage = "Nuevo proyecto '{$input['nombre']}' ha sido creado.";
        $notificationLink = "tareas.php?project_id={$input['id']}";

        foreach ($users as $user) {
            $userRole = $user['role'] ?? 'team_member';
            $userEmail = $user['email'] ?? null;

            // Notify admins and project managers, but not the creator
            if ($userEmail && $userEmail !== $creatorEmail) {
                if ($userRole === 'admin' || $userRole === 'project_manager') {
                    createNotification('../data/notifications.json', $userEmail, $notificationMessage, $notificationLink);
                }
            }
        }

        http_response_code(201);
        echo json_encode($input);
        break;

    case 'PUT':
        checkPermission('project_manager'); // --- PERMISSION CHECK ---
        if (!$id) {
            http_response_code(400);
            echo json_encode(['error' => 'Project ID is required']);
            exit;
        }

        $input = json_decode(file_get_contents('php://input'), true);
        if (!$input) {
            http_response_code(400);
            echo json_encode(['error' => 'Invalid JSON']);
            exit;
        }

        $projects = getProjects($filePath);
        $projectFound = false;
        $updatedProject = null;
        foreach ($projects as $key => $project) {
            if ($project['id'] === $id) {
                $input['budget'] = $input['budget'] ?? $project['budget'] ?? 0;
                $projects[$key] = array_merge($project, $input);
                $projects[$key]['id'] = $id;
                $updatedProject = $projects[$key];
                $projectFound = true;
                break;
            }
        }

        if ($projectFound) {
            saveProjects($filePath, $projects);
            // --- Log Activity ---
            logActivity($id, "actualizó los detalles del proyecto '{$updatedProject['nombre']}'.");

            echo json_encode($updatedProject);
        } else {
            http_response_code(404);
            echo json_encode(['error' => 'Project not found']);
        }
        break;

    case 'DELETE':
        checkPermission('project_manager'); // --- PERMISSION CHECK ---
        if (!$id) {
            http_response_code(400);
            echo json_encode(['error' => 'Project ID is required']);
            exit;
        }

        $projects = getProjects($filePath);
        $projectToDelete = null;
        foreach ($projects as $project) {
            if ($project['id'] === $id) {
                $projectToDelete = $project;
                break;
            }
        }

        $initialCount = count($projects);
        $projects = array_filter($projects, function($project) use ($id) {
            return $project['id'] !== $id;
        });

        if (count($projects) < $initialCount && $projectToDelete) {
            logActivity($id, "eliminó el proyecto '{$projectToDelete['nombre']}'.");
            saveProjects($filePath, array_values($projects));
            http_response_code(204);
        } else {
            http_response_code(404);
            echo json_encode(['error' => 'Project not found']);
        }
        break;

    default:
        http_response_code(405);
        echo json_encode(['error' => 'Method Not Allowed']);
        break;
}
?>