<?php
session_start();
header('Content-Type: application/json');

// --- PERMISSION CHECK FUNCTION (Copied from usuarios.php) ---
function checkPermission($requiredRole) {
    if (!isset($_SESSION['user_role'])) {
        http_response_code(401); // Unauthorized
        echo json_encode(['error' => 'No autenticado.']);
        exit();
    }

    $userRole = $_SESSION['user_role'];
    $rolesHierarchy = [
        'admin' => 3,
        'project_manager' => 2,
        'team_member' => 1
    ];

    if (!isset($rolesHierarchy[$userRole]) || !isset($rolesHierarchy[$requiredRole])) {
        http_response_code(500); // Internal Server Error - invalid role configured
        echo json_encode(['error' => 'Error de configuración de roles.']);
        exit();
    }

    if ($rolesHierarchy[$userRole] < $rolesHierarchy[$requiredRole]) {
        http_response_code(403); // Forbidden
        echo json_encode(['error' => 'Permiso denegado. Rol insuficiente.']);
        exit();
    }
}
// --- END PERMISSION CHECK FUNCTION ---

checkPermission('project_manager'); // Apply permission check for GET operation

$projectsFilePath = '../data/projects.json';
$tasksFilePath = '../data/tasks.json';
$usersFilePath = '../data/usuarios.json';
$expensesFilePath = '../data/expenses.json';

$project_id = isset($_GET['project_id']) ? $_GET['project_id'] : null;

if (!$project_id) {
    http_response_code(400);
    echo json_encode(['error' => 'ID de proyecto es requerido']);
    exit;
}

function getData($filePath) {
    if (!file_exists($filePath) || filesize($filePath) === 0) {
        return [];
    }
    $fileContent = file_get_contents($filePath);
    $data = json_decode($fileContent, true);
    return json_last_error() === JSON_ERROR_NONE ? $data : [];
}

$projects = getData($projectsFilePath);
$tasks = getData($tasksFilePath);
$users = getData($usersFilePath);
$expenses = getData($expensesFilePath);

$project = array_filter($projects, function($p) use ($project_id) {
    return $p['id'] === $project_id;
});
$project = reset($project); // Get the first (and only) matching project

if (!$project) {
    http_response_code(404);
    echo json_encode(['error' => 'Proyecto no encontrado']);
    exit;
}

$projectBudget = $project['budget'] ?? 0;

// Calculate work log costs
$totalWorkLogCost = 0;
$userHourlyRates = [];
foreach ($users as $user) {
    $userHourlyRates[$user['email']] = $user['hourly_rate'] ?? 0;
}

foreach ($tasks as $task) {
    if (isset($task['project_id']) && $task['project_id'] === $project_id && isset($task['trabajos_realizados'])) {
        foreach ($task['trabajos_realizados'] as $log) {
            if (isset($task['responsible_email']) && isset($log['hours'])) {
                $hourlyRate = $userHourlyRates[$task['responsible_email']] ?? 0;
                $totalWorkLogCost += ($log['hours'] * $hourlyRate);
            }
        }
    }
}

// Calculate total expenses
$totalExpenses = 0;
$projectExpenses = array_filter($expenses, function($expense) use ($project_id) {
    return $expense['project_id'] === $project_id;
});
foreach ($projectExpenses as $expense) {
    $totalExpenses += $expense['amount'] ?? 0;
}

$totalCost = $totalWorkLogCost + $totalExpenses;
$balance = $projectBudget - $totalCost;

echo json_encode([
    'project_id' => $project_id,
    'budget' => $projectBudget,
    'total_work_log_cost' => $totalWorkLogCost,
    'total_expenses' => $totalExpenses,
    'total_cost' => $totalCost,
    'balance' => $balance
]);
?>