<?php
session_start();
header('Content-Type: application/json');

// --- PERMISSION CHECK FUNCTION (Copied from usuarios.php) ---
function checkPermission($requiredRole) {
    if (!isset($_SESSION['user_role'])) {
        http_response_code(401); // Unauthorized
        echo json_encode(['error' => 'No autenticado.']);
        exit();
    }

    $userRole = $_SESSION['user_role'];
    $rolesHierarchy = [
        'admin' => 3,
        'project_manager' => 2,
        'team_member' => 1
    ];

    if (!isset($rolesHierarchy[$userRole]) || !isset($rolesHierarchy[$requiredRole])) {
        http_response_code(500); // Internal Server Error - invalid role configured
        echo json_encode(['error' => 'Error de configuración de roles.']);
        exit();
    }

    if ($rolesHierarchy[$userRole] < $rolesHierarchy[$requiredRole]) {
        http_response_code(403); // Forbidden
        echo json_encode(['error' => 'Permiso denegado. Rol insuficiente.']);
        exit();
    }
}
// --- END PERMISSION CHECK FUNCTION ---

$filePath = '../data/expenses.json';

if (!file_exists($filePath) || filesize($filePath) === 0) {
    file_put_contents($filePath, '[]');
}

$method = $_SERVER['REQUEST_METHOD'];
$id = isset($_GET['id']) ? $_GET['id'] : null;
$project_id = isset($_GET['project_id']) ? $_GET['project_id'] : null;

function getExpenses($filePath) {
    $fileContent = file_get_contents($filePath);
    $expenses = json_decode($fileContent, true);
    return json_last_error() === JSON_ERROR_NONE ? $expenses : [];
}

function saveExpenses($filePath, $expenses) {
    $fileHandle = fopen($filePath, 'w');
    if (flock($fileHandle, LOCK_EX)) {
        fwrite($fileHandle, json_encode($expenses, JSON_PRETTY_PRINT));
        flock($fileHandle, LOCK_UN);
    }
    fclose($fileHandle);
}

switch ($method) {
    case 'GET':
        $expenses = getExpenses($filePath);
        if ($project_id) {
            $expenses = array_filter($expenses, function($expense) use ($project_id) {
                return $expense['project_id'] === $project_id;
            });
            $expenses = array_values($expenses);
        }
        echo json_encode($expenses);
        break;

    case 'POST':
        checkPermission('project_manager'); // --- PERMISSION CHECK ---
        $input = json_decode(file_get_contents('php://input'), true);
        if (!$input || !isset($input['project_id']) || !isset($input['description']) || !isset($input['amount']) || !isset($input['date'])) {
            http_response_code(400);
            echo json_encode(['error' => 'Datos incompletos para el gasto']);
            exit;
        }

        $expenses = getExpenses($filePath);
        $input['id'] = 'exp_' . uniqid();
        $expenses[] = $input;
        saveExpenses($filePath, $expenses);

        http_response_code(201);
        echo json_encode($input);
        break;

    case 'PUT':
        checkPermission('project_manager'); // --- PERMISSION CHECK ---
        if (!$id) {
            http_response_code(400);
            echo json_encode(['error' => 'ID de gasto es requerido']);
            exit;
        }

        $input = json_decode(file_get_contents('php://input'), true);
        if (!$input) {
            http_response_code(400);
            echo json_encode(['error' => 'Invalid JSON']);
            exit;
        }

        $expenses = getExpenses($filePath);
        $expenseFound = false;
        $updatedExpense = null;
        foreach ($expenses as $key => $expense) {
            if ($expense['id'] === $id) {
                $expenses[$key] = array_merge($expense, $input);
                $expenses[$key]['id'] = $id;
                $updatedExpense = $expenses[$key];
                $expenseFound = true;
                break;
            }
        }

        if ($expenseFound) {
            saveExpenses($filePath, $expenses);
            echo json_encode($updatedExpense);
        } else {
            http_response_code(404);
            echo json_encode(['error' => 'Gasto no encontrado']);
        }
        break;

    case 'DELETE':
        checkPermission('project_manager'); // --- PERMISSION CHECK ---
        if (!$id) {
            http_response_code(400);
            echo json_encode(['error' => 'ID de gasto es requerido']);
            exit;
        }

        $expenses = getExpenses($filePath);
        $initialCount = count($expenses);
        $expenses = array_filter($expenses, function($expense) use ($id) {
            return $expense['id'] !== $id;
        });

        if (count($expenses) < $initialCount) {
            saveExpenses($filePath, array_values($expenses));
            http_response_code(204);
        } else {
            http_response_code(404);
            echo json_encode(['error' => 'Gasto no encontrado']);
        }
        break;

    default:
        http_response_code(405);
        echo json_encode(['error' => 'Method Not Allowed']);
        break;
}
?>