<?php
session_start();
header('Content-Type: application/json');

// --- PERMISSION CHECK FUNCTION (Copied from usuarios.php) ---
function checkPermission($requiredRole) {
    if (!isset($_SESSION['user_role'])) {
        http_response_code(401); // Unauthorized
        echo json_encode(['error' => 'No autenticado.']);
        exit();
    }

    $userRole = $_SESSION['user_role'];
    $rolesHierarchy = [
        'admin' => 3,
        'project_manager' => 2,
        'team_member' => 1
    ];

    if (!isset($rolesHierarchy[$userRole]) || !isset($rolesHierarchy[$requiredRole])) {
        http_response_code(500); // Internal Server Error - invalid role configured
        echo json_encode(['error' => 'Error de configuración de roles.']);
        exit();
    }

    if ($rolesHierarchy[$userRole] < $rolesHierarchy[$requiredRole]) {
        http_response_code(403); // Forbidden
        echo json_encode(['error' => 'Permiso denegado. Rol insuficiente.']);
        exit();
    }
}
// --- END PERMISSION CHECK FUNCTION ---

checkPermission('team_member'); // Apply permission check for all operations in this API

function find_private_conversation($user1_email, $user2_email, $conversations_file) {
    if (!file_exists($conversations_file)) {
        return null;
    }
    $conversations = json_decode(file_get_contents($conversations_file), true);
    foreach ($conversations as $conv) {
        if ($conv['type'] === 'private') {
            $participants = $conv['participants'];
            if (count($participants) === 2 && in_array($user1_email, $participants) && in_array($user2_email, $participants)) {
                return $conv;
            }
        }
    }
    return null;
}

$conversations_file = '../data/conversations.json';
$messages_file = '../data/messages.json'; // Path to messages data
$users_file = '../data/usuarios.json';

function get_conversations($user_email, $conversations_file) {
    if (!file_exists($conversations_file)) {
        return [];
    }

    $conversations = json_decode(file_get_contents($conversations_file), true);
    $user_conversations = [];

    foreach ($conversations as $conv) {
        if (in_array($user_email, $conv['participants'])) {
            $user_conversations[] = $conv;
        }
    }

    return $user_conversations;
}

function saveJsonData($filePath, $data) {
    $fileHandle = fopen($filePath, 'w');
    if (flock($fileHandle, LOCK_EX)) {
        fwrite($fileHandle, json_encode($data, JSON_PRETTY_PRINT));
        flock($fileHandle, LOCK_UN);
    }
    fclose($fileHandle);
}

switch ($_SERVER['REQUEST_METHOD']) {
    case 'GET':
        if (isset($_GET['with_user'])) {
            $existing_conversation = find_private_conversation($_SESSION['user_email'], $_GET['with_user'], $conversations_file);
            if ($existing_conversation) {
                echo json_encode($existing_conversation);
            } else {
                http_response_code(404);
                echo json_encode(['error' => 'No se encontró la conversación']);
            }
        } else {
            echo json_encode(get_conversations($_SESSION['user_email'], $conversations_file));
        }
        break;

    case 'POST':
        $data = json_decode(file_get_contents('php://input'), true);

        if (!isset($data['type']) || !isset($data['participants'])) {
            http_response_code(400);
            echo json_encode(['error' => 'Datos incompletos']);
            exit();
        }

        $conversations = file_exists($conversations_file) ? json_decode(file_get_contents($conversations_file), true) : [];

        // For private chats, check if a conversation already exists
        if ($data['type'] === 'private' && count($data['participants']) === 1) {
            $other_user = $data['participants'][0];
            $existing_conversation = find_private_conversation($_SESSION['user_email'], $other_user, $conversations_file);
            if ($existing_conversation) {
                echo json_encode($existing_conversation);
                exit();
            }
        }

        $new_conversation = [
            'id' => uniqid('conv_'),
            'type' => $data['type'],
            'participants' => $data['participants'],
            'name' => isset($data['name']) ? htmlspecialchars($data['name']) : null,
            'creator' => $_SESSION['user_email'],
            'created_at' => date('Y-m-d H:i:s'),
            'foto' => $data['foto'] ?? null // Add foto field
        ];

        if (!in_array($_SESSION['user_email'], $new_conversation['participants'])) {
            $new_conversation['participants'][] = $_SESSION['user_email'];
        }

        $conversations[] = $new_conversation;
        saveJsonData($conversations_file, $conversations);
        echo json_encode($new_conversation);
        break;

    case 'PUT':
        checkPermission('project_manager'); // Only project_manager or admin can edit groups
        $id = $_GET['id'] ?? null;
        if (!$id) {
            http_response_code(400);
            echo json_encode(['error' => 'Conversation ID is required']);
            exit();
        }

        $input = json_decode(file_get_contents('php://input'), true);
        if (!$input) {
            http_response_code(400);
            echo json_encode(['error' => 'Invalid JSON']);
            exit();
        }

        $conversations = file_exists($conversations_file) ? json_decode(file_get_contents($conversations_file), true) : [];
        $conversationFound = false;
        $updatedConversation = null;

        foreach ($conversations as $key => $conv) {
            if ($conv['id'] === $id) {
                // Only allow editing of group chats, or private chats if specific fields are updated
                if ($conv['type'] === 'group') {
                    $conversations[$key]['name'] = isset($input['name']) ? htmlspecialchars($input['name']) : $conv['name'];
                    $conversations[$key]['participants'] = $input['participants'] ?? $conv['participants'];
                    $conversations[$key]['foto'] = $input['foto'] ?? $conv['foto'] ?? null; // Update foto field

                    // Ensure current user is still a participant if they were before
                    if (!in_array($_SESSION['user_email'], $conversations[$key]['participants'])) {
                        $conversations[$key]['participants'][] = $_SESSION['user_email'];
                    }

                    $updatedConversation = $conversations[$key];
                    $conversationFound = true;
                    break;
                } else if ($conv['type'] === 'private') {
                    // For private chats, only allow specific updates if needed, e.g., foto
                    // For now, we'll just allow foto update for private chats too if sent
                    $conversations[$key]['foto'] = $input['foto'] ?? $conv['foto'] ?? null;
                    $updatedConversation = $conversations[$key];
                    $conversationFound = true;
                    break;
                }
            }
        }

        if ($conversationFound) {
            saveJsonData($conversations_file, $conversations);
            echo json_encode($updatedConversation);
        } else {
            http_response_code(404);
            echo json_encode(['error' => 'Conversation not found']);
        }
        break;

    case 'DELETE':
        checkPermission('project_manager'); // Only project_manager or admin can delete groups
        $id = $_GET['id'] ?? null;
        if (!$id) {
            http_response_code(400);
            echo json_encode(['error' => 'Conversation ID is required']);
            exit();
        }

        $conversations = file_exists($conversations_file) ? json_decode(file_get_contents($conversations_file), true) : [];
        $messages = file_exists($messages_file) ? json_decode(file_get_contents($messages_file), true) : [];

        $initialConversationsCount = count($conversations);
        $conversations = array_filter($conversations, function($conv) use ($id) {
            return $conv['id'] !== $id;
        });

        $initialMessagesCount = count($messages);
        $messages = array_filter($messages, function($msg) use ($id) {
            return $msg['conversation_id'] !== $id;
        });

        if (count($conversations) < $initialConversationsCount) {
            saveJsonData($conversations_file, array_values($conversations));
            saveJsonData($messages_file, array_values($messages)); // Save updated messages
            http_response_code(204); // No Content
        } else {
            http_response_code(404);
            echo json_encode(['error' => 'Conversation not found']);
        }
        break;

    default:
        http_response_code(405);
        echo json_encode(['error' => 'Method Not Allowed']);
        break;
}
?>