<?php
session_start();
header('Content-Type: application/json');

// --- PERMISSION CHECK FUNCTION (Copied from usuarios.php) ---
function checkPermission($requiredRole) {
    if (!isset($_SESSION['user_role'])) {
        http_response_code(401); // Unauthorized
        echo json_encode(['error' => 'No autenticado.']);
        exit();
    }

    $userRole = $_SESSION['user_role'];
    $rolesHierarchy = [
        'admin' => 3,
        'project_manager' => 2,
        'team_member' => 1
    ];

    if (!isset($rolesHierarchy[$userRole]) || !isset($rolesHierarchy[$requiredRole])) {
        http_response_code(500); // Internal Server Error - invalid role configured
        echo json_encode(['error' => 'Error de configuración de roles.']);
        exit();
    }

    if ($rolesHierarchy[$userRole] < $rolesHierarchy[$requiredRole]) {
        http_response_code(403); // Forbidden
        echo json_encode(['error' => 'Permiso denegado. Rol insuficiente.']);
        exit();
    }
}
// --- END PERMISSION CHECK FUNCTION ---

require_once 'activity_log_helper.php';
require_once 'notification_helper.php';
$filePath = '../data/comments.json';

// Ensure the file exists and is a valid JSON array
if (!file_exists($filePath) || filesize($filePath) === 0) {
    file_put_contents($filePath, '[]');
}

$method = $_SERVER['REQUEST_METHOD'];
$taskId = isset($_GET['task_id']) ? $_GET['task_id'] : null;

function getComments($filePath) {
    $fileContent = file_get_contents($filePath);
    $comments = json_decode($fileContent, true);
    return json_last_error() === JSON_ERROR_NONE ? $comments : [];
}

function saveComments($filePath, $comments) {
    $fileHandle = fopen($filePath, 'w');
    if (flock($fileHandle, LOCK_EX)) {
        fwrite($fileHandle, json_encode($comments, JSON_PRETTY_PRINT));
        flock($fileHandle, LOCK_UN);
    }
    fclose($fileHandle);
}

switch ($method) {
    case 'GET':
        $comments = getComments($filePath);
        if ($taskId) {
            $filteredComments = array_filter($comments, function($comment) use ($taskId) {
                return isset($comment['task_id']) && $comment['task_id'] === $taskId;
            });
            echo json_encode(array_values($filteredComments)); // Re-index array
        } else {
            echo json_encode($comments);
        }
        break;

    case 'POST':
        checkPermission('team_member'); // --- PERMISSION CHECK ---

        $taskId = $_POST['task_id'] ?? null;
        $commentText = $_POST['comment_text'] ?? null;

        if (!$taskId || !$commentText || empty(trim($commentText))) {
            http_response_code(400);
            echo json_encode(['error' => 'Datos incompletos para el comentario.']);
            exit;
        }

        $attachmentPath = null;
        if (isset($_FILES['attachment']) && $_FILES['attachment']['error'] === UPLOAD_ERR_OK) {
            $uploadDir = '../uploads/comments/';
            if (!is_dir($uploadDir)) {
                mkdir($uploadDir, 0777, true);
            }
            $fileName = uniqid('file_') . '_' . basename($_FILES['attachment']['name']);
            $targetPath = $uploadDir . $fileName;

            if (move_uploaded_file($_FILES['attachment']['tmp_name'], $targetPath)) {
                // Store a relative path accessible from the web root
                $attachmentPath = 'uploads/comments/' . $fileName;
            } else {
                http_response_code(500);
                echo json_encode(['error' => 'Error al mover el archivo subido.']);
                exit;
            }
        }

        $comments = getComments($filePath);

        $newComment = [
            'id' => uniqid('comment_'),
            'task_id' => $taskId,
            'user_email' => $_SESSION['user_email'],
            'user_name' => $_SESSION['user_name'],
            'comment_text' => htmlspecialchars(trim($commentText)),
            'timestamp' => date('Y-m-d H:i:s'),
            'attachment' => $attachmentPath
        ];

        $comments[] = $newComment;
        saveComments($filePath, $comments);

        // --- Log Activity ---
        $allTasks = json_decode(file_get_contents('../data/tasks.json'), true) ?: [];
        $taskName = 'desconocida';
        $projectId = null;
        foreach ($allTasks as $task) {
            if ($task['id'] === $taskId) {
                $taskName = $task['name'];
                $projectId = $task['project_id'];
                break;
            }
        }
        if ($projectId) {
            logActivity($projectId, "añadió un comentario a la tarea '{$taskName}'.");
        }

        // --- Handle @mentions and send notifications ---
        preg_match_all('/@([\p{L}\p{N}\s_.-]+?)\b/u', $commentText, $matches);
        $mentionedNames = array_unique($matches[1]);

        if (!empty($mentionedNames)) {
            $allUsers = json_decode(file_get_contents('../data/usuarios.json'), true) ?: [];
            $commenterName = $_SESSION['user_name'];
            $notificationLink = "tareas.php?project_id={$projectId}";

            foreach ($mentionedNames as $name) {
                foreach ($allUsers as $user) {
                    if (trim($name) === $user['nombreCompleto'] && $user['email'] !== $_SESSION['user_email']) {
                        $notificationMessage = "{$commenterName} te ha mencionado en un comentario en la tarea '{$taskName}'.";
                        createNotification('../data/notifications.json', $user['email'], $notificationMessage, $notificationLink);
                    }
                }
            }
        }

        http_response_code(201);
        echo json_encode($newComment);
        break;

    default:
        http_response_code(405);
        echo json_encode(['error' => 'Method Not Allowed']);
        break;
}
?>