<?php
session_start();
header('Content-Type: application/json');

// --- PERMISSION CHECK FUNCTION (Copied from usuarios.php) ---
function checkPermission($requiredRole) {
    if (!isset($_SESSION['user_role'])) {
        http_response_code(401); // Unauthorized
        echo json_encode(['error' => 'No autenticado.']);
        exit();
    }

    $userRole = $_SESSION['user_role'];
    $rolesHierarchy = [
        'admin' => 3,
        'project_manager' => 2,
        'team_member' => 1
    ];

    if (!isset($rolesHierarchy[$userRole]) || !isset($rolesHierarchy[$requiredRole])) {
        http_response_code(500); // Internal Server Error - invalid role configured
        echo json_encode(['error' => 'Error de configuración de roles.']);
        exit();
    }

    if ($rolesHierarchy[$userRole] < $rolesHierarchy[$requiredRole]) {
        http_response_code(403); // Forbidden
        echo json_encode(['error' => 'Permiso denegado. Rol insuficiente.']);
        exit();
    }
}
// --- END PERMISSION CHECK FUNCTION ---

$filePath = '../data/clients.json';

if (!file_exists($filePath) || filesize($filePath) === 0) {
    file_put_contents($filePath, '[]');
}

$method = $_SERVER['REQUEST_METHOD'];
$id = isset($_GET['id']) ? $_GET['id'] : null;

function getItems($filePath) {
    $fileContent = file_get_contents($filePath);
    $items = json_decode($fileContent, true);
    return json_last_error() === JSON_ERROR_NONE ? $items : [];
}

function saveItems($filePath, $items) {
    $fileHandle = fopen($filePath, 'w');
    if (flock($fileHandle, LOCK_EX)) {
        fwrite($fileHandle, json_encode($items, JSON_PRETTY_PRINT));
        flock($fileHandle, LOCK_UN);
    }
    fclose($fileHandle);
}

switch ($method) {
    case 'GET':
        echo json_encode(getItems($filePath));
        break;

    case 'POST':
        checkPermission('project_manager'); // --- PERMISSION CHECK ---
        $input = json_decode(file_get_contents('php://input'), true);
        if (!$input) { http_response_code(400); echo json_encode(['error' => 'Invalid JSON']); exit; }

        $items = getItems($filePath);
        $input['id'] = 'client_' . time();
        $items[] = $input;
        saveItems($filePath, $items);

        http_response_code(201);
        echo json_encode($input);
        break;

    case 'PUT':
        checkPermission('project_manager'); // --- PERMISSION CHECK ---
        if (!$id) { http_response_code(400); echo json_encode(['error' => 'ID is required']); exit; }
        $input = json_decode(file_get_contents('php://input'), true);
        if (!$input) { http_response_code(400); echo json_encode(['error' => 'Invalid JSON']); exit; }

        $items = getItems($filePath);
        $found = false;
        $updatedItem = null;
        foreach ($items as $key => $item) {
            if ($item['id'] === $id) {
                $items[$key] = array_merge($item, $input);
                $items[$key]['id'] = $id;
                $updatedItem = $items[$key];
                $found = true;
                break;
            }
        }

        if ($found) {
            saveItems($filePath, $items);
            echo json_encode($updatedItem);
        } else {
            http_response_code(404);
            echo json_encode(['error' => 'Item not found']);
        }
        break;

    case 'DELETE':
        checkPermission('project_manager'); // --- PERMISSION CHECK ---
        if (!$id) { http_response_code(400); echo json_encode(['error' => 'ID is required']); exit; }

        $items = getItems($filePath);
        $initialCount = count($items);
        $items = array_filter($items, function($item) use ($id) {
            return $item['id'] !== $id;
        });

        if (count($items) < $initialCount) {
            saveItems($filePath, array_values($items));
            http_response_code(204);
        } else {
            http_response_code(404);
            echo json_encode(['error' => 'Item not found']);
        }
        break;

    default:
        http_response_code(405);
        echo json_encode(['error' => 'Method Not Allowed']);
        break;
}
?>