<?php
session_start();
header('Content-Type: application/json');

// --- PERMISSION CHECK FUNCTION (Copied from usuarios.php) ---
function checkPermission($requiredRole) {
    if (!isset($_SESSION['user_role'])) {
        http_response_code(401); // Unauthorized
        echo json_encode(['error' => 'No autenticado.']);
        exit();
    }

    $userRole = $_SESSION['user_role'];
    $rolesHierarchy = [
        'admin' => 3,
        'project_manager' => 2,
        'team_member' => 1
    ];

    if (!isset($rolesHierarchy[$userRole]) || !isset($rolesHierarchy[$requiredRole])) {
        http_response_code(500); // Internal Server Error - invalid role configured
        echo json_encode(['error' => 'Error de configuración de roles.']);
        exit();
    }

    if ($rolesHierarchy[$userRole] < $rolesHierarchy[$requiredRole]) {
        http_response_code(403); // Forbidden
        echo json_encode(['error' => 'Permiso denegado. Rol insuficiente.']);
        exit();
    }
}
// --- END PERMISSION CHECK FUNCTION ---

$messages_file = '../data/messages.json';
$conversations_file = '../data/conversations.json';
$users_file = '../data/usuarios.json'; // Path to users data

// Function to get JSON data from a file
function getJsonData($filePath) {
    if (!file_exists($filePath) || filesize($filePath) === 0) {
        return [];
    }
    $fileContent = file_get_contents($filePath);
    $data = json_decode($fileContent, true);
    return json_last_error() === JSON_ERROR_NONE ? $data : [];
}

// Function to save messages (renamed from saveComments)
function saveMessages($filePath, $messages) {
    $fileHandle = fopen($filePath, 'w');
    if (flock($fileHandle, LOCK_EX)) {
        fwrite($fileHandle, json_encode($messages, JSON_PRETTY_PRINT));
        flock($fileHandle, LOCK_UN);
    }
    fclose($fileHandle);
}

// Include notification helper function
require_once 'notification_helper.php';

function getConversationParticipants($convId, $convFile) {
    $conversations = getJsonData($convFile);
    foreach ($conversations as $conv) {
        if ($conv['id'] === $convId) {
            return $conv['participants'] ?? [];
        }
    }
    return [];
}

if ($_SERVER['REQUEST_METHOD'] === 'GET') {
    checkPermission('team_member'); // --- PERMISSION CHECK ---
    if (!isset($_GET['conversation_id'])) {
        http_response_code(400);
        echo json_encode(['error' => 'Falta el ID de la conversación']);
        exit();
    }

    $conversation_id = $_GET['conversation_id'];
    $messages = getJsonData($messages_file);
    $users = getJsonData($users_file); // Load users data

    // Create a map of user email to user photo
    $userPhotos = [];
    foreach ($users as $user) {
        $userPhotos[$user['email']] = $user['foto'] ?? 'https://via.placeholder.com/50'; // Default photo
    }
    
    $conversation_messages = [];
    foreach ($messages as $message) {
        if ($message['conversation_id'] === $conversation_id) {
            // Add sender_photo to the message object
            $message['sender_photo'] = $userPhotos[$message['sender_email']] ?? 'https://via.placeholder.com/50';
            $conversation_messages[] = $message;
        }
    }

    echo json_encode($conversation_messages);
}

if ($_SERVER['REQUEST_METHOD'] === 'POST') {
    checkPermission('team_member'); // --- PERMISSION CHECK ---

    $data = json_decode(file_get_contents('php://input'), true);

    if (isset($data['message']) && !empty(trim($data['message'])) && isset($data['conversation_id'])) {
        $messages = getJsonData($messages_file);

        $new_message = [
            'id' => uniqid(),
            'conversation_id' => $data['conversation_id'],
            'sender_email' => $_SESSION['user_email'],
            'sender_name' => $_SESSION['user_name'],
            'message' => htmlspecialchars(trim($data['message'])),
            'timestamp' => date('Y-m-d H:i:s')
        ];

        $messages[] = $new_message;
        saveMessages($messages_file, $messages); // Corrected function call

        // --- Create Notifications ---
        $participants = getConversationParticipants($data['conversation_id'], $conversations_file);
        $senderName = $_SESSION['user_name'];
        $notificationMessage = "Nuevo mensaje de {$senderName}";
        $notificationLink = "chat.php"; // Link to the main chat page

        foreach ($participants as $participantEmail) {
            // Don't notify the sender
            if ($participantEmail !== $_SESSION['user_email']) {
                createNotification('../data/notifications.json', $participantEmail, $notificationMessage, $notificationLink);
            }
        }

        http_response_code(201);
        echo json_encode($new_message);
    } else {
        http_response_code(400);
        echo json_encode(['error' => 'Datos incompletos']);
    }
}
?>