<?php
// Función para leer la base de datos
function get_data() {
    $database_file = 'database.json';
    if (!file_exists($database_file)) return null;
    return json_decode(file_get_contents($database_file), true);
}

// Función para formatear números usando la configuración global
function format_number_php($value, $config) {
    $decimals = !empty($config['cantidad_decimales']) ? intval($config['cantidad_decimales']) : 0;
    $dec_point = !empty($config['separador_decimales']) ? $config['separador_decimales'] : ',';
    $thousands_sep = !empty($config['separador_miles']) ? $config['separador_miles'] : '.';
    $currency_symbol = !empty($config['simbolo_moneda']) ? $config['simbolo_moneda'] : 'G.';
    return $currency_symbol . ' ' . number_format($value, $decimals, $dec_point, $thousands_sep);
}

// Obtener parámetros de la URL
$start_date_str = $_GET['start_date'] ?? null;
$end_date_str = $_GET['end_date'] ?? null;

if (!$start_date_str || !$end_date_str) {
    die('Error: Debe especificar una fecha de inicio y una fecha de fin.');
}

$start_date = strtotime($start_date_str);
$end_date = strtotime($end_date_str . ' 23:59:59');

$data = get_data();
$config = $data['configuracion'] ?? []; // Cargar configuración
$ingresos_data = $data['financiero'] ?? [];
$egresos_data = $data['egresos'] ?? [];

// 1. Calcular Ingresos Totales y IVA
$total_ingresos = 0;
$total_iva = 0;
$ingresos_periodo = array_filter($ingresos_data, function($ing) use ($start_date, $end_date) {
    $fecha_ingreso = strtotime($ing['fecha']);
    return $fecha_ingreso >= $start_date && $fecha_ingreso <= $end_date && $ing['estado'] === 'Pagada';
});

foreach ($ingresos_periodo as $ingreso) {
    $monto = floatval($ingreso['monto_total']);
    $total_ingresos += $monto;
    if ($ingreso['iva_tipo'] === '10') {
        $total_iva += $monto - ($monto / 1.1);
    } elseif ($ingreso['iva_tipo'] === '5') {
        $total_iva += $monto - ($monto / 1.05);
    }
}

// 2. Calcular Egresos Totales
$total_egresos = 0;
$egresos_periodo = array_filter($egresos_data, function($eg) use ($start_date, $end_date) {
    $fecha_egreso = strtotime($eg['fecha']);
    return $fecha_egreso >= $start_date && $fecha_egreso <= $end_date;
});
foreach ($egresos_periodo as $egreso) {
    $total_egresos += floatval($egreso['monto_total']);
}

// 3. Calcular Resultado Final (Balance)
$resultado_clinica = $total_ingresos - $total_egresos;

?>
<!DOCTYPE html>
<html lang="es">
<head>
    <meta charset="UTF-8">
    <title>Informe Financiero de la Clínica</title>
    <script src="https://cdn.jsdelivr.net/npm/chart.js"></script>
    <style>
        body { font-family: -apple-system, BlinkMacSystemFont, "Segoe UI", Roboto, "Helvetica Neue", Arial, sans-serif; line-height: 1.6; color: #333; background-color: #f4f4f4; margin: 0; padding: 0; }
        .page { width: 90%; max-width: 1200px; padding: 2cm; margin: 1cm auto; border: 1px #D3D3D3 solid; border-radius: 5px; background: white; box-shadow: 0 0 5px rgba(0, 0, 0, 0.1); }
        .toolbar { text-align: center; padding: 10px; background-color: #333; }
        .toolbar button { padding: 10px 20px; font-size: 16px; cursor: pointer; background-color: #3498db; color: white; border: none; border-radius: 5px; }
        .report-header { text-align: center; border-bottom: 2px solid #3498db; padding-bottom: 15px; margin-bottom: 30px; }
        .report-header h1 { margin: 0; font-size: 2em; }
        .report-header p { color: #555; }
        .summary-grid { display: grid; grid-template-columns: repeat(auto-fit, minmax(200px, 1fr)); gap: 20px; margin-bottom: 30px; }
        .summary-item { padding: 20px; border-radius: 5px; text-align: center; }
        .summary-item h4 { margin: 0 0 10px 0; font-size: 1rem; }
        .summary-item p { margin: 0; font-size: 1.6em; font-weight: bold; }
        .ingresos { background-color: #e8f5e9; color: #2e7d32; }
        .costos, .egresos { background-color: #ffebee; color: #c62828; }
        .iva { background-color: #fff3e0; color: #ef6c00; }
        .resultado { background-color: #e3f2fd; color: #1565c0; }
        .chart-container { width: 100%; max-width: 700px; margin: 40px auto; }
        @media print {
            body, .page { margin: 0; box-shadow: none; border: none; width: 100%; }
            .toolbar { display: none; }
            .page { padding: 1cm; }
        }
    </style>
</head>
<body>
    <div class="toolbar">
        <button onclick="window.print()">Imprimir o Guardar como PDF</button>
    </div>
    <div class="page">
        <div class="report-header">
            <h1>Informe Financiero de la Clínica</h1>
            <p>
                Período del <strong><?php echo date('d/m/Y', $start_date); ?></strong> 
                al <strong><?php echo date('d/m/Y', $end_date); ?></strong>
            </p>
        </div>

        <div class="summary-grid">
            <div class="summary-item ingresos">
                <h4>(+) Ingresos Totales</h4>
                <p><?php echo format_number_php($total_ingresos, $config); ?></p>
            </div>
            <div class="summary-item egresos">
                <h4>(-) Gastos de Clínica</h4>
                <p><?php echo format_number_php($total_egresos, $config); ?></p>
            </div>
            <div class="summary-item iva">
                <h4>(=) IVA a Pagar</h4>
                <p><?php echo format_number_php($total_iva, $config); ?></p>
            </div>
            <div class="summary-item resultado">
                <h4>(=) Resultado Final</h4>
                <p><?php echo format_number_php($resultado_clinica, $config); ?></p>
            </div>
        </div>

        <div class="chart-container">
            <canvas id="financialChart"></canvas>
        </div>

    </div>

    <script>
        const ctx = document.getElementById('financialChart').getContext('2d');
        const financialChart = new Chart(ctx, {
            type: 'bar',
            data: {
                labels: ['Ingresos', 'Egresos'],
                datasets: [{
                    label: 'Monto',
                    data: [
                        <?php echo $total_ingresos; ?>,
                        <?php echo $total_egresos; ?>
                    ],
                    backgroundColor: [
                        'rgba(39, 174, 96, 0.6)',
                        'rgba(192, 57, 43, 0.6)'
                    ],
                    borderColor: [
                        'rgba(39, 174, 96, 1)',
                        'rgba(231, 76, 60, 1)',
                        'rgba(192, 57, 43, 1)'
                    ],
                    borderWidth: 1
                }]
            },
            options: {
                indexAxis: 'y', // Hace el gráfico de barras horizontal
                responsive: true,
                plugins: {
                    legend: {
                        display: false
                    },
                    title: {
                        display: true,
                        text: 'Resumen de Ingresos y Egresos'
                    }
                },
                scales: {
                    x: {
                        beginAtZero: true,
                        title: {
                            display: true,
                            text: 'Monto'
                        }
                    }
                }
            }
        });
    </script>
</body>
</html>