document.addEventListener('DOMContentLoaded', () => {
    const PACIENTES_API_URL = 'api.php?entity=pacientes';
    const PROFESIONALES_API_URL = 'api.php?entity=profesionales';
    const CONFIG_API_URL = 'api.php?entity=configuracion';

    // Elementos del DOM
    const pacienteForm = document.getElementById('paciente-form');
    const tableBody = document.getElementById('pacientes-table-body');
    const modal = document.getElementById('paciente-modal');
    const openModalBtn = document.getElementById('open-modal-btn');
    const closeBtn = document.querySelector('.close-btn');
    const searchInput = document.getElementById('search-input');
    const tableHead = document.querySelector('#pacientes-table thead');
    let allPacientes = [];
    let allProfesionales = [];

    // Inicializar el input de teléfono para pacientes
    const telefonoInput = document.querySelector("#telefono");
    const iti = window.intlTelInput(telefonoInput, {
        initialCountry: "py", // Paraguay por defecto
        separateDialCode: true,
        utilsScript: "https://cdnjs.cloudflare.com/ajax/libs/intl-tel-input/17.0.8/js/utils.js",
    });

    // --- Control del Modal ---
    const openModal = () => modal.style.display = "block";
    const closeModal = () => {
        modal.style.display = "none";
        pacienteForm.reset();
        document.getElementById('paciente-id').value = '';
    };

    openModalBtn.onclick = () => {
        populateProfesionalesDropdown(); // Rellenar el desplegable para un nuevo paciente
        openModal();                     // Y luego abrir el modal
    };
    closeBtn.onclick = closeModal;
    window.onclick = (event) => {
        if (event.target == modal) {
            closeModal();
        }
    }

    // --- Lógica de la Aplicación ---

    const populateProfesionalesDropdown = () => {
        const select = document.getElementById('profesional_asignado_id');
        select.innerHTML = '<option value="">-- Asignar Profesional --</option>'; // Limpiar y añadir opción por defecto
        allProfesionales.forEach(prof => {
            const option = document.createElement('option');
            option.value = prof.id;
            option.textContent = `${prof.nombre} ${prof.apellido}`;
            select.appendChild(option);
        });
    };

    const renderTable = (pacientes) => {
        tableBody.innerHTML = '';
        pacientes.forEach(p => {
            // Buscar el nombre del profesional usando el ID guardado
            const profesionalAsignado = allProfesionales.find(prof => prof.id === p.profesional_asignado_id);
            const nombreProfesional = profesionalAsignado ? `${profesionalAsignado.nombre} ${profesionalAsignado.apellido}` : 'N/A';

            // Preparar enlace de WhatsApp
            const cleanPhoneNumber = (p.telefono || '').replace(/\D/g, '');
            const whatsappLink = cleanPhoneNumber ? `https://wa.me/${cleanPhoneNumber}` : '#';
            const whatsappIcon = cleanPhoneNumber ? `<a href="${whatsappLink}" target="_blank" class="whatsapp-icon" title="Enviar WhatsApp"><svg xmlns="http://www.w3.org/2000/svg" width="18" height="18" viewBox="0 0 24 24" fill="#25D366"><path d="M19.7,4.3c-1.9-1.9-4.4-2.9-7.1-2.9c-5.6,0-10.1,4.5-10.1,10.1c0,1.8,0.5,3.5,1.3,5l-1.5,5.5l5.6-1.5 c1.4,0.8,3,1.3,4.7,1.3h0c5.6,0,10.1-4.5,10.1-10.1C22.6,8.7,21.6,6.2,19.7,4.3z M12.6,20.5h0c-1.6,0-3.2-0.5-4.5-1.4l-0.3-0.2 l-3.4,0.9l0.9-3.3l-0.2-0.3c-0.9-1.4-1.5-3-1.5-4.7c0-4.5,3.6-8.1,8.1-8.1c2.2,0,4.3,0.9,5.8,2.4c1.5,1.5,2.4,3.5,2.4,5.8 C20.7,16.9,17.1,20.5,12.6,20.5z M18.2,14.8c-0.3-0.1-1.5-0.7-1.7-0.8c-0.2-0.1-0.4-0.1-0.6,0.1c-0.2,0.2-0.6,0.8-0.8,0.9 c-0.1,0.1-0.3,0.2-0.5,0.1c-0.2-0.1-1-0.4-1.9-1.2c-0.7-0.6-1.2-1.4-1.3-1.6c-0.1-0.2,0-0.4,0.1-0.5c0.1-0.1,0.2-0.3,0.4-0.4 c0.1-0.1,0.2-0.2,0.2-0.4c0.1-0.1,0-0.3-0.1-0.4c-0.1-0.1-0.6-1.5-0.8-2c-0.2-0.5-0.4-0.5-0.6-0.5c-0.2,0-0.4,0-0.6,0 c-0.2,0-0.5,0.1-0.8,0.4c-0.3,0.3-1,1-1,2.5c0,1.5,1,2.9,1.2,3.1c0.2,0.2,2,3.2,4.9,4.3c0.7,0.3,1.2,0.4,1.6,0.5 c0.7,0.2,1.3,0.1,1.8-0.1c0.5-0.2,1.5-0.6,1.7-1.2c0.2-0.6,0.2-1.1,0.1-1.2C18.6,14.9,18.4,14.9,18.2,14.8z"></path></svg></a>` : '';

            const row = document.createElement('tr');
            row.innerHTML = `
                <td data-label="Nombre">${p.nombre || ''}</td>
                <td data-label="Apellido">${p.apellido || ''}</td>
                <td data-label="Teléfono">${p.telefono || ''} ${whatsappIcon}</td>
                <td data-label="Motivo">${(p.motivo_consulta || '').substring(0, 30)}...</td>
                <td data-label="Profesional">${nombreProfesional}</td>
                <td data-label="Estado"><span class="status status-${p.estado || 'inactivo'}">${p.estado || 'inactivo'}</span></td>
                <td data-label="Acciones" class="actions-cell">
                    <a href="paciente_detalle.php?id=${p.id}" class="action-btn view-btn">Ver Ficha</a>
                    <button class="edit-btn" data-id="${p.id}">Editar</button>
                    <button class="delete-btn" data-id="${p.id}">Eliminar</button>
                </td>
            `;
            tableBody.appendChild(row);
        });
    };

    const fetchAllDataAndRender = async () => {
        // Cargar ambas listas de datos en paralelo
        const [pacientesRes, profesionalesRes] = await Promise.all([
            fetch(PACIENTES_API_URL),
            fetch(PROFESIONALES_API_URL)
        ]);
        allPacientes = await pacientesRes.json();
        allProfesionales = await profesionalesRes.json();

        allPacientes.sort((a, b) => (a.nombre || '').localeCompare(b.nombre || ''));
        renderTable(allPacientes);
        searchInput.value = '';
    };

    pacienteForm.addEventListener('submit', async (e) => {
        e.preventDefault();
        const id = document.getElementById('paciente-id').value;
        
        const pacienteData = {
            nombre: document.getElementById('nombre').value,
            apellido: document.getElementById('apellido').value,
            dni: document.getElementById('dni').value,
            fecha_nacimiento: document.getElementById('fecha_nacimiento').value,
            telefono: iti.getNumber(), // Obtener el número completo
            email: document.getElementById('email').value,
            profesional_asignado_id: document.getElementById('profesional_asignado_id').value,
            estado: document.getElementById('estado').value,
            motivo_consulta: document.getElementById('motivo_consulta').value,
            antecedentes: document.getElementById('antecedentes').value,
        };

        let url = PACIENTES_API_URL;
        if (id) {
            url = `${PACIENTES_API_URL}&id=${id}`;
        }
        const response = await fetch(url, {
            method: 'POST',
            headers: { 'Content-Type': 'application/json' },
            body: JSON.stringify(pacienteData)
        });

        const result = await response.json();
        console.log(result.message);

        closeModal();
        fetchAllDataAndRender();
    });

    searchInput.addEventListener('input', (e) => {
        const searchTerm = e.target.value.toLowerCase();
        const filtered = allPacientes.filter(p => {
            const fullName = `${p.nombre || ''} ${p.apellido || ''}`.toLowerCase();
            return fullName.includes(searchTerm);
        });
        renderTable(filtered);
    });

    let sortState = { key: 'nombre', order: 'asc' };
    tableHead.addEventListener('click', (e) => {
        const target = e.target;
        if (target.tagName === 'TH' && target.dataset.sort) {
            const sortKey = target.dataset.sort;
            const order = (sortState.key === sortKey && sortState.order === 'asc') ? 'desc' : 'asc';

            allPacientes.sort((a, b) => {
                const valA = (a[sortKey] || '').toLowerCase();
                const valB = (b[sortKey] || '').toLowerCase();
                return order === 'asc' ? valA.localeCompare(valB) : valB.localeCompare(valA);
            });

            sortState = { key: sortKey, order: order };
            renderTable(allPacientes);
        }
    });

    tableBody.addEventListener('click', async (e) => {
        const target = e.target;
        const id = target.dataset.id;

        if (target.classList.contains('delete-btn')) {
            if (confirm('¿Estás seguro de que quieres eliminar a este paciente?')) {
                const response = await fetch(`${PACIENTES_API_URL}&id=${id}`, {
                    method: 'DELETE'
                });
                const result = await response.json();
                console.log(result.message);
                fetchAllDataAndRender();
            }
        }

        if (target.classList.contains('edit-btn')) {
            const pacienteAEditar = allPacientes.find(p => p.id === id);
            if (pacienteAEditar) {
                document.getElementById('paciente-id').value = pacienteAEditar.id;
                document.getElementById('nombre').value = pacienteAEditar.nombre || '';
                document.getElementById('apellido').value = pacienteAEditar.apellido || '';
                document.getElementById('dni').value = pacienteAEditar.dni || '';
                document.getElementById('fecha_nacimiento').value = pacienteAEditar.fecha_nacimiento || '';
                iti.setNumber(pacienteAEditar.telefono || ''); // Establecer el número
                document.getElementById('email').value = pacienteAEditar.email || '';
                document.getElementById('estado').value = pacienteAEditar.estado || 'activo';
                document.getElementById('motivo_consulta').value = pacienteAEditar.motivo_consulta || '';
                document.getElementById('antecedentes').value = pacienteAEditar.antecedentes || '';
                
                populateProfesionalesDropdown(); // Llenar el dropdown
                document.getElementById('profesional_asignado_id').value = pacienteAEditar.profesional_asignado_id || '';
                openModal();
            }
        }
    });

    fetchAllDataAndRender();
});