document.addEventListener('DOMContentLoaded', () => {
    // --- URLs de la API ---
    const CITAS_API_URL = 'api.php?entity=citas';
    const PACIENTES_API_URL = 'api.php?entity=pacientes';
    const PROFESIONALES_API_URL = 'api.php?entity=profesionales';
    const CONFIG_API_URL = 'api.php?entity=configuracion';

    // --- Elementos del DOM ---
    const modal = document.getElementById('cita-modal');
    const openModalBtn = document.getElementById('open-modal-btn');
    const closeBtn = document.querySelector('.close-btn');
    const citaForm = document.getElementById('cita-form');
    const tableBody = document.getElementById('citas-table-body');
    const searchInput = document.getElementById('search-input');

    // Nuevos elementos para la vista de calendario
    const viewButtons = document.querySelectorAll('.view-btn');
    const calendarView = document.getElementById('calendar-view');
    const listView = document.getElementById('list-view');
    const calendarTitle = document.getElementById('calendar-title');
    const prevBtn = document.getElementById('prev-btn');
    const nextBtn = document.getElementById('next-btn');

    // --- Almacenamiento de datos ---
    let allCitas = [];
    let allPacientes = [];
    let allProfesionales = [];

    // --- Estado de la UI ---
    let currentView = 'week'; // 'week' o 'list'
    let currentDate = new Date(); // Fecha para navegar por el calendario

    // --- Control del Modal ---
    const openModal = () => modal.style.display = "block";
    const closeModal = () => {
        modal.style.display = "none";
        citaForm.reset();
        document.getElementById('cita-id').value = '';
    };

    openModalBtn.onclick = openModal;
    closeBtn.onclick = closeModal;
    window.onclick = (event) => {
        if (event.target == modal) closeModal();
    };

    // --- Lógica de la Aplicación ---

    const populateDropdowns = () => {
        const pacienteSelect = document.getElementById('paciente_id');
        const profesionalSelect = document.getElementById('profesional_id');

        pacienteSelect.innerHTML = '<option value="">-- Seleccionar Paciente --</option>';
        allPacientes.forEach(p => {
            pacienteSelect.innerHTML += `<option value="${p.id}">${p.nombre} ${p.apellido}</option>`;
        });

        profesionalSelect.innerHTML = '<option value="">-- Seleccionar Profesional --</option>';
        allProfesionales.forEach(p => {
            profesionalSelect.innerHTML += `<option value="${p.id}">${p.nombre} ${p.apellido}</option>`;
        });
    };

    const renderListView = (citas) => {
        tableBody.innerHTML = '';
        citas.forEach(cita => {
            const paciente = allPacientes.find(p => p.id === cita.paciente_id);
            const profesional = allProfesionales.find(p => p.id === cita.profesional_id);

            const row = document.createElement('tr');
            row.innerHTML = `
                <td data-label="Fecha y Hora">${new Date(cita.fecha + 'T' + cita.hora).toLocaleString('es-ES', { day: '2-digit', month: '2-digit', year: 'numeric', hour: '2-digit', minute: '2-digit' })}</td>
                <td data-label="Paciente">${paciente ? `${paciente.nombre} ${paciente.apellido}` : 'N/A'}</td>
                <td data-label="Profesional">${profesional ? `${profesional.nombre} ${profesional.apellido}` : 'N/A'}</td>
                <td data-label="Estado"><span class="status status-cita-${cita.estado.toLowerCase()}">${cita.estado}</span></td>
                <td data-label="Acciones" class="actions-cell">
                    <button class="edit-btn" data-id="${cita.id}">Editar</button>
                    <button class="delete-btn" data-id="${cita.id}">Eliminar</button>
                </td>
            `;
            tableBody.appendChild(row);
        });
    };

    const renderDayWeekView = () => {
        calendarView.innerHTML = '';
        calendarView.className = currentView === 'day' ? 'day-view' : 'week-view';
        calendarView.style.gridTemplateColumns = currentView === 'day' ? '1fr' : 'repeat(7, 1fr)';

        const range = getVisibleDateRange();
        const weekStart = range.start;
        const weekEnd = range.end;
        const numDays = (currentView === 'day') ? 1 : 7;

        const options = { month: 'long', day: 'numeric' };
        if (currentView === 'day') {
            calendarTitle.textContent = `${weekStart.toLocaleDateString('es-ES', { dateStyle: 'full' })}`;
        } else {
            calendarTitle.textContent = `${weekStart.toLocaleDateString('es-ES', options)} - ${weekEnd.toLocaleDateString('es-ES', options)}, ${weekStart.getFullYear()}`;
        }

        for (let i = 0; i < numDays; i++) {
            const day = new Date(weekStart);
            day.setDate(weekStart.getDate() + i);

            const dayColumn = document.createElement('div');
            dayColumn.className = 'calendar-day' + (isToday(day) ? ' today' : '');
            dayColumn.dataset.date = day.toISOString().split('T')[0]; // YYYY-MM-DD
            dayColumn.innerHTML = `<div class="calendar-day-header">${day.toLocaleDateString('es-ES', { weekday: 'long', day: 'numeric' })}</div>`;

            // Filtrar y ordenar citas para este día
            const dayCitas = allCitas
                .filter(cita => {
                    const citaDate = new Date(cita.fecha + 'T00:00:00'); // Corregido para evitar problemas de timezone
                    return citaDate.getFullYear() === day.getFullYear() &&
                           citaDate.getMonth() === day.getMonth() &&
                           citaDate.getDate() === day.getDate();
                })
                .sort((a, b) => a.hora.localeCompare(b.hora));

            dayCitas.forEach(cita => {
                const paciente = allPacientes.find(p => p.id === cita.paciente_id);
                const profesional = allProfesionales.find(p => p.id === cita.profesional_id);
                const color = generateColorFromString(profesional ? profesional.id : '');
                const hueMatch = color.match(/hsl\((\d+)/);
                const borderColor = hueMatch ? `hsl(${Math.abs(hueMatch[1])}, 70%, 65%)` : 'hsl(200, 70%, 65%)'; // Un poco más oscuro para el borde

                const citaElement = document.createElement('div');
                citaElement.className = 'calendar-appointment';
                citaElement.dataset.id = cita.id;
                citaElement.draggable = true;
                citaElement.style.setProperty('--appointment-bg-color', color);
                citaElement.style.setProperty('--appointment-border-color', borderColor);
                citaElement.innerHTML = `
                    <p class="time">${cita.hora.substring(0, 5)}</p>
                    <p>${paciente ? `${paciente.nombre} ${paciente.apellido}` : 'Paciente no encontrado'}</p>
                `;
                dayColumn.appendChild(citaElement);
            });

            calendarView.appendChild(dayColumn);
        }
    };

    const renderMonthView = () => {
        calendarView.innerHTML = '';
        calendarView.className = 'month-view';
        calendarView.style.gridTemplateColumns = 'repeat(7, 1fr)';

        const range = getVisibleDateRange();
        const monthStart = range.start;
        const monthEnd = range.end;

        calendarTitle.textContent = `${currentDate.toLocaleDateString('es-ES', { month: 'long', year: 'numeric' })}`;

        // Encabezados de los días de la semana
        const weekdays = ['Lun', 'Mar', 'Mié', 'Jue', 'Vie', 'Sáb', 'Dom'];
        weekdays.forEach(wd => {
            calendarView.innerHTML += `<div class="calendar-day-header">${wd}</div>`;
        });

        // Días del calendario
        let currentDay = new Date(monthStart);
        while (currentDay <= monthEnd) {
            const dayCell = document.createElement('div');
            dayCell.className = 'month-day';
            if (currentDay.getMonth() !== currentDate.getMonth()) {
                dayCell.classList.add('other-month');
            }
            if (isToday(currentDay)) {
                dayCell.classList.add('today');
            }
            dayCell.dataset.date = currentDay.toISOString().split('T')[0]; // YYYY-MM-DD
            dayCell.innerHTML = `<div class="day-number">${currentDay.getDate()}</div>`;

            // Añadir citas
            const dayCitas = allCitas
                .filter(cita => {
                    const citaDate = new Date(cita.fecha + 'T00:00:00'); // Corregido para evitar problemas de timezone
                    return citaDate.getFullYear() === currentDay.getFullYear() &&
                           citaDate.getMonth() === currentDay.getMonth() &&
                           citaDate.getDate() === currentDay.getDate();
                })
                .sort((a, b) => a.hora.localeCompare(b.hora));

            dayCitas.forEach(cita => {
                const paciente = allPacientes.find(p => p.id === cita.paciente_id);
                const profesional = allProfesionales.find(p => p.id === cita.profesional_id);
                const color = generateColorFromString(profesional ? profesional.id : '');

                const citaElement = document.createElement('div');
                citaElement.className = 'month-appointment';
                citaElement.dataset.id = cita.id;
                citaElement.draggable = true;
                citaElement.style.setProperty('--appointment-bg-color', color);
                citaElement.title = `${cita.hora.substring(0, 5)} - ${paciente ? paciente.nombre : ''} (${profesional ? profesional.nombre : 'N/A'})`;
                citaElement.textContent = `${cita.hora} ${paciente ? paciente.nombre.split(' ')[0] : ''}`;
                dayCell.appendChild(citaElement);
            });

            calendarView.appendChild(dayCell);
            currentDay.setDate(currentDay.getDate() + 1);
        }
    };

    const renderCurrentView = () => {
        // Ocultar todas las vistas primero
        calendarView.style.display = 'none';
        listView.style.display = 'none';

        if (currentView === 'list') {
            listView.style.display = 'block';
            const searchTerm = searchInput.value.toLowerCase();
            const filteredCitas = allCitas.filter(cita => {
                const paciente = allPacientes.find(p => p.id === cita.paciente_id);
                const profesional = allProfesionales.find(p => p.id === cita.profesional_id);
                const pacienteNombre = paciente ? `${paciente.nombre} ${paciente.apellido}`.toLowerCase() : '';
                const profesionalNombre = profesional ? `${profesional.nombre} ${profesional.apellido}`.toLowerCase() : '';
                return pacienteNombre.includes(searchTerm) || profesionalNombre.includes(searchTerm);
            });
            filteredCitas.sort((a, b) => new Date(b.fecha + 'T' + b.hora) - new Date(a.fecha + 'T' + a.hora));
            renderListView(filteredCitas);
        } else { // Vistas de calendario
            calendarView.style.display = 'grid';
            renderDayWeekView();
            renderMonthView();
        }
    };

    const fetchAllDataAndRender = async () => {
        try {
            const [citasRes, pacientesRes, profesionalesRes, configRes] = await Promise.all([
                fetch(CITAS_API_URL),
                fetch(PACIENTES_API_URL),
                fetch(PROFESIONALES_API_URL),
                fetch(CONFIG_API_URL)
            ]);

            allCitas = await citasRes.json();
            allPacientes = await pacientesRes.json();
            allProfesionales = await profesionalesRes.json();
            const config = await configRes.json();

            // Renderizar la vista actual
            renderCurrentView();
            populateDropdowns();

        } catch (error) {
            console.error("Error al cargar los datos:", error);
        }
    };

    citaForm.addEventListener('submit', async (e) => {
        e.preventDefault();
        const id = document.getElementById('cita-id').value;

        const citaData = {
            paciente_id: document.getElementById('paciente_id').value,
            profesional_id: document.getElementById('profesional_id').value,
            fecha: document.getElementById('fecha').value,
            hora: document.getElementById('hora').value,
            estado: document.getElementById('estado').value,
            notas: document.getElementById('notas').value,
        };

        let url = CITAS_API_URL;
        if (id) {
            url = `${CITAS_API_URL}&id=${id}`;
        }

        await fetch(url, {
            method: 'POST',
            headers: { 'Content-Type': 'application/json' },
            body: JSON.stringify(citaData)
        });

        closeModal();
        await fetchAllDataAndRender();
    });

    const handleEditOrDelete = async (e) => {
        const target = e.target;
        const id = target.dataset.id;

        if (target.classList.contains('delete-btn')) {
            if (confirm('¿Estás seguro de que quieres eliminar esta cita?')) {
                await fetch(`${CITAS_API_URL}&id=${id}`, { method: 'DELETE' });
                await fetchAllDataAndRender();
            }
        }

        if (target.classList.contains('edit-btn')) {
            const citaAEditar = allCitas.find(c => c.id === id);
            if (citaAEditar) {
                document.getElementById('cita-id').value = citaAEditar.id;
                document.getElementById('paciente_id').value = citaAEditar.paciente_id;
                document.getElementById('profesional_id').value = citaAEditar.profesional_id;
                document.getElementById('fecha').value = citaAEditar.fecha;
                document.getElementById('hora').value = citaAEditar.hora;
                document.getElementById('estado').value = citaAEditar.estado;
                document.getElementById('notas').value = citaAEditar.notas || '';
                openModal();

                // --- Lógica para mostrar el historial ---
                const historyContainer = document.getElementById('cita-history-container');
                const historyList = document.getElementById('cita-history-list');
                historyList.innerHTML = '';

                if (citaAEditar.history && citaAEditar.history.length > 0) {
                    historyContainer.style.display = 'block';
                    citaAEditar.history.forEach(log => {
                        const listItem = document.createElement('li');
                        const changesText = log.changes.join(', ');
                        listItem.innerHTML = `
                            ${changesText}
                            <span class="log-meta">Por ${log.user_name} el ${new Date(log.timestamp).toLocaleString('es-ES')}</span>
                        `;
                        historyList.appendChild(listItem);
                    });
                } else {
                    historyContainer.style.display = 'none';
                }
            }
        }
    };

    // Añadir listeners a ambas vistas
    tableBody.addEventListener('click', handleEditOrDelete);
    calendarView.addEventListener('click', (e) => {
        let target = e.target.closest('.calendar-appointment');
        if (!target) {
            target = e.target.closest('.month-appointment');
        }

        if (target) {
            // Simular un evento para reutilizar la lógica de edición
            handleEditOrDelete({ target: { classList: { contains: () => true }, dataset: { id: target.dataset.id } } });
        }
    });

    // --- Lógica de Arrastrar y Soltar (Drag and Drop) ---
    let draggedAppointmentId = null;

    calendarView.addEventListener('dragstart', (e) => {
        const target = e.target.closest('.calendar-appointment, .month-appointment');
        if (target) {
            draggedAppointmentId = target.dataset.id;
            // Usamos setTimeout para permitir que el navegador "capture" la imagen del elemento antes de ocultarlo
            setTimeout(() => {
                target.classList.add('dragging');
            }, 0);
        }
    });

    calendarView.addEventListener('dragend', (e) => {
        const target = e.target.closest('.calendar-appointment, .month-appointment');
        if (target) {
            target.classList.remove('dragging');
        }
        draggedAppointmentId = null;
    });

    calendarView.addEventListener('dragover', (e) => {
        e.preventDefault(); // Esencial para permitir el evento 'drop'
        const dropTarget = e.target.closest('.calendar-day, .month-day');
        if (dropTarget) {
            // Añadir un indicador visual
            document.querySelectorAll('.drag-over').forEach(el => el.classList.remove('drag-over'));
            dropTarget.classList.add('drag-over');
        }
    });

    calendarView.addEventListener('dragleave', (e) => {
        const dropTarget = e.target.closest('.calendar-day, .month-day');
        if (dropTarget) {
            dropTarget.classList.remove('drag-over');
        }
    });

    calendarView.addEventListener('drop', async (e) => {
        e.preventDefault();
        document.querySelectorAll('.drag-over').forEach(el => el.classList.remove('drag-over'));

        const dropTarget = e.target.closest('.calendar-day, .month-day');
        if (!dropTarget || !draggedAppointmentId) return;

        const newDate = dropTarget.dataset.date;
        const citaToUpdate = allCitas.find(c => c.id === draggedAppointmentId);

        if (citaToUpdate && citaToUpdate.fecha !== newDate) {
            citaToUpdate.fecha = newDate; // Actualización local
            renderCurrentView(); // Re-renderizar inmediatamente para feedback visual

            // Actualización en el servidor
            await fetch(`${CITAS_API_URL}&id=${draggedAppointmentId}`, { method: 'POST', headers: { 'Content-Type': 'application/json' }, body: JSON.stringify({ fecha: newDate }) });
        }
    });

    searchInput.addEventListener('input', (e) => {
        // La búsqueda solo tiene sentido en la vista de lista por ahora
        if (currentView === 'list') {
            renderCurrentView();
        }
    });

    viewButtons.forEach(button => {
        button.addEventListener('click', () => {
            currentView = button.dataset.view;
            viewButtons.forEach(btn => btn.classList.remove('active'));
            button.classList.add('active');
            renderCurrentView();
        });
    });

    const navigate = (direction) => {
        switch (currentView) {
            case 'day':
                currentDate.setDate(currentDate.getDate() + direction);
                break;
            case 'week':
                currentDate.setDate(currentDate.getDate() + (direction * 7));
                break;
            case 'month':
                currentDate.setMonth(currentDate.getMonth() + direction);
                break;
        }
        renderCurrentView();
    };

    prevBtn.addEventListener('click', () => navigate(-1));
    nextBtn.addEventListener('click', () => navigate(1));

    // --- Funciones de Utilidad ---
    const getVisibleDateRange = () => {
        const d = new Date(currentDate);
        if (currentView === 'day') {
            return { start: d, end: d };
        }
        if (currentView === 'week') {
            const day = d.getDay();
            const diff = d.getDate() - day + (day === 0 ? -6 : 1); // Lunes como primer día
            const start = new Date(d.setDate(diff));
            const end = new Date(start);
            end.setDate(start.getDate() + 6);
            return { start, end };
        }
        if (currentView === 'month') {
            const startOfMonth = new Date(d.getFullYear(), d.getMonth(), 1);
            const endOfMonth = new Date(d.getFullYear(), d.getMonth() + 1, 0);

            // Extender para completar la semana
            const startDay = startOfMonth.getDay();
            const startDiff = (startDay === 0 ? -6 : 1 - startDay);
            const start = new Date(startOfMonth); // Clonar la fecha
            start.setDate(start.getDate() + startDiff);

            const endDay = endOfMonth.getDay();
            const endDiff = (endDay === 0 ? 0 : 7 - endDay);
            const end = new Date(endOfMonth); // Clonar la fecha
            end.setDate(end.getDate() + endDiff);

            return { start, end };
        }
    };

    const isToday = (date) => {
        const today = new Date();
        return date.getDate() === today.getDate() &&
               date.getMonth() === today.getMonth() &&
               date.getFullYear() === today.getFullYear();
    };

    /**
     * Genera un color HSL pastel consistente a partir de una cadena (ej. ID de profesional).
     * @param {string} str La cadena para generar el color.
     * @returns {string} Un color en formato hsl(hue, saturation, lightness).
     */
    const generateColorFromString = (str) => {
        let hash = 0;
        if (!str || str.length === 0) return 'hsl(200, 50%, 90%)'; // Color por defecto
        for (let i = 0; i < str.length; i++) {
            hash = str.charCodeAt(i) + ((hash << 5) - hash);
            hash = hash & hash; // Convertir a entero de 32 bits
        }
        const hue = Math.abs(hash % 360);
        return `hsl(${hue}, 70%, 85%)`; // Tono variable, saturación y luminosidad fijas para colores pastel
    };

    // Carga inicial de todos los datos
    fetchAllDataAndRender();
});